--------------------------------------- zentrale BDE-Funktionen Anfang -------------------------------------

--
CREATE OR REPLACE FUNCTION TPersonal.bde__stempeln__execute(
      _minr                   integer,                    -- Mitarbeiter-Nr.
      _stempel_zeitpunkt      timestamp = currenttime(),  -- Stempelzeit (Kommen oder Gehen)
      _abw_id                 integer = null,             -- Abwesenheit (Gehen mit Abwesenheit)
      _terminal_info          varchar = null,             -- Terminal-Infos (Geräte-Nr, SN usw.)
      _net_ip                 varchar = null,             -- IP des Terminals

      OUT bde__status_before  TPersonal.bde__status,      -- Status vorm Kommen/Gehen
      OUT bde__status_after   TPersonal.bde__status       -- Status nach Kommen/Gehen
  ) RETURNS record AS $$
  BEGIN
      -- zentrale Funktion für Prozesse der An- bzw. Abstempelung


      -- Fehler: Ohne Angabe von Mitarbeiter oder Stempelzeit
      IF ( _minr IS NULL OR _stempel_zeitpunkt IS NULL ) THEN
          RAISE EXCEPTION '%', lang_text( 16833 );
      END IF;


      -- aktuellen Status der Anwesenheit bzw. Abwesenheit vor Prozess Kommen oder Gehen ermitteln (ohne Summen)
      bde__status_before :=
          TPersonal.bde__status__get(
              _minr,
              _cur_state_only => true,
              _target_time    => _stempel_zeitpunkt
          )
      ;

      -- In Abhängigkeit vom aktuellen Status (vor Prozess) wird auszuführender Prozess Kommen oder Gehen entschieden.
          -- Abwesend
          IF NOT bde__status_before.bs_is_present THEN
              -- Kommen
              -- ggf. kommen aus aktueller Abwesenheit (cur_abw_id), z.B. aus Raucherpause, Pause
              PERFORM TPersonal.bde__stempeln__start__execute( _minr, _stempel_zeitpunkt, bde__status_before.bs_abw_id, _terminal_info, _net_ip );

          -- Anwesend
          ELSE
              -- Gehen
              -- ggf. gehen mit angg. Abwesenheit
              PERFORM TPersonal.bde__stempeln__ende__execute( _minr, _stempel_zeitpunkt, _abw_id, _terminal_info, _net_ip );

          END IF;
      -- Ende vom ausgeführten Prozess Kommen oder Gehen

      --  Status der Anwesenheit bzw. Abwesenheit nach Prozess Kommen oder Gehen ermitteln (inkl. Summen)
      bde__status_after :=
          TPersonal.bde__status__get(
              _minr,
              _cur_state_only => false,
              _target_time    => _stempel_zeitpunkt
          )
      ;


      RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;
--

-- gem. Wiki Der Aufruf ist nie direkt und muss immer über TPersonal.bde__stempeln__execute durchgeführt werden, da erst der BDE-Status ermittelt wird.
-- https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/BDE-Funktionen_zentral
CREATE OR REPLACE FUNCTION TPersonal.bde__stempeln__start__execute(
      _minr               integer,                    -- Mitarbeiter-Nr.
      _stempel_zeitpunkt  timestamp = currenttime(),  -- Stempelzeit (Kommen)
      _cur_abw_id         integer = null,             -- ggf. kommen aus aktueller Abwesenheit
      _terminal_info      varchar = null,             -- Terminal-Infos (Geräte-Nr, SN usw.)
      _net_ip             varchar = null              -- IP des Terminals
  ) RETURNS void AS $$
  DECLARE
      _stempel_date     CONSTANT date := _stempel_zeitpunkt::date;
      _stempel_wt       CONSTANT date := coalesce( TPersonal.bdep__individwt__get_date__by_bd_anf_minr( _stempel_zeitpunkt, _minr )::date, _stempel_date );

      _id_raucherpause  CONSTANT integer := 103; -- feste ID der Raucherpause
      _id_pause         CONSTANT integer := 110; -- feste ID der Pause
  BEGIN
      -- zentrale Funktion für Anstempeln
        -- Aufruf immer über TPersonal.bde__stempeln__execute, da erst Präsenz-Status (bs_is_present) ermittelt wird.
        -- inkl. erneutes Anstempeln aus spez. Abwesenheiten
        -- inkl. Anstempeln unterbrochener Aufträge durch Abwesenheiten


      -- Fehler: Ohne Angabe von Mitarbeiter oder Stempelzeit
      IF ( _minr IS NULL OR _stempel_zeitpunkt IS NULL ) THEN
          RAISE EXCEPTION '%', lang_text( 16833 );
      END IF;


      -- Präsenzzeit eintragen
      IF  -- Kommen Standard
              _cur_abw_id IS NULL
          -- Kommen aus Abwesenheiten (Pause, Stundenausgleich, ...)
          OR  _cur_abw_id NOT IN ( _id_raucherpause )
      THEN
          INSERT INTO bdep ( bd_minr,  bd_anf,             bd_stemphint,   bd_terminal )
          VALUES           ( _minr,    _stempel_zeitpunkt, _terminal_info, _net_ip )
          ;

      -- Sonderfall Kommen aus Raucherpause (103)
      -- Präsenzzeit ist noch offen und Eintrag von Raucherpause in Abwesenheiten mit Zeitstempel ist vorhanden.
      ELSIF   _cur_abw_id = _id_raucherpause THEN
          UPDATE bdepab SET
            bdab_end      = _stempel_date,
            bdab_endt     = _stempel_zeitpunkt::time
          WHERE bdab_minr = _minr
            AND bdab_aus_id = _id_raucherpause
            AND bdab_endt IS NULL
            -- ggf. Raucherpause über Mitternacht.
            AND bdab_anf BETWEEN _stempel_wt AND _stempel_date
          ;

      END IF;


      -- unterbrochene Aufträge wieder anstempeln
      -- Nur durchführen, wenn die Abweichung der einzutragenden Stempelzeit +-5 Minuten zu jetzt ist (Stichwort: verzögerte Datenübertragung, falsche Uhrzeit).
      IF ( currenttime() - _stempel_zeitpunkt ) BETWEEN '-5 minutes'::interval AND '+5 minutes'::interval THEN
          -- erneuter Start der mit Abwesenheit gekennzeichneten Aufträge
          INSERT INTO bdea ( ba_anf,             ba_minr, ba_ix, ba_op, ba_ks, ba_ksap, ba_ruest, ba_id_interrupt_last, ba_txt )
          SELECT             _stempel_zeitpunkt, ba_minr, ba_ix, ba_op, ba_ks, ba_ksap, ba_ruest, ba_id               , ''
          FROM bdea
          WHERE ba_minr = _minr
            AND NOT ba_ende
            -- Auftrag wurde mit Abwesenheit (Raucherpause, Pause) unterbrochen
            AND ba_aus_id IN ( _id_raucherpause, _id_pause )
            -- timestamp_to_date(bd_anf) geht über Index, daher kein CAST.
            -- veraltet: Bei Pause war es 7 Tage rückwirkend: (ba_aus_id = 110 AND ba_anf::date > current_date - 7). Grund unklar.
            AND timestamp_to_date( ba_anf ) BETWEEN _stempel_wt AND _stempel_date
            -- Auftrag ist nicht bereits angestempelt, vgl. Trigger allow_insert_bdea
            AND NOT EXISTS (
                SELECT true
                FROM bdea AS bdea_trg
                WHERE bdea_trg.ba_minr  = bdea.ba_minr
                  AND bdea_trg.ba_ix    = bdea.ba_ix
                  AND bdea_trg.ba_op    = bdea.ba_op
                  AND bdea_trg.ba_ks    = bdea.ba_ks
                  AND bdea_trg.ba_end     IS NULL
                  AND bdea_trg.ba_efftime IS NULL
              )
          ;

          -- aktuelle Einträge der Abwesenheiten (Raucherpause, Pause) entfernen
          UPDATE bdea SET
            ba_aus_id = null
          WHERE ba_minr = _minr
            -- AND NOT ba_ende
            AND ba_end IS NOT NULL
            AND ba_aus_id IN ( _id_raucherpause, _id_pause )
            -- timestamp_to_date(bd_anf) geht über Index, daher kein CAST.
            AND timestamp_to_date( ba_anf ) BETWEEN _stempel_wt AND _stempel_date
          ;

      END IF;


      RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;
--

-- gem. Wiki Der Aufruf ist nie direkt und muss immer über TPersonal.bde__stempeln__execute durchgeführt werden, da erst der BDE-Status ermittelt wird.
-- https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/BDE-Funktionen_zentral
CREATE OR REPLACE FUNCTION TPersonal.bde__stempeln__ende__execute(
      _minr               integer,                    -- Mitarbeiter-Nr.
      _stempel_zeitpunkt  timestamp = currenttime(),  -- Stempelzeit (Gehen)
      _abw_id             integer = null,             -- ggf. gehen mit Abwesenheit
      _terminal_info      varchar = null,             -- Terminal-Infos (Geräte-Nr, SN usw.)
      _net_ip             varchar = null              -- IP des Terminals
  ) RETURNS void AS $$
  DECLARE
      _stempel_date     CONSTANT date := _stempel_zeitpunkt::date;

      _id_raucherpause  CONSTANT integer := 103; -- feste ID der Raucherpause
      _id_pause         CONSTANT integer := 110; -- feste ID der Pause
  BEGIN
      -- zentrale Funktion für Abstempeln
        -- Aufruf immer über TPersonal.bde__stempeln__execute, da erst Präsenz-Status (bs_is_present) ermittelt wird.


      -- Fehler: Ohne Angabe von Mitarbeiter oder Stempelzeit
      IF ( _minr IS NULL OR _stempel_zeitpunkt IS NULL ) THEN
          RAISE EXCEPTION '%', lang_text( 16833 );
      END IF;


      -- Präsenzzeit-Ende eintragen
      IF  -- Gehen Standard
              _abw_id IS NULL
          -- Gehen mit Abwesenheiten (Pause, Krank usw. außer Raucherpause)
          OR  _abw_id NOT IN ( _id_raucherpause )
      THEN
          UPDATE bdep SET
            bd_end        = _stempel_zeitpunkt,
            bd_aus_id     = _abw_id,
            bd_stemphint  = concat_ws( ';', nullif( trim( bd_stemphint ), '' ), _terminal_info )::varchar(50),
            bd_terminal   = concat_ws( ';', nullif( trim( bd_terminal  ), '' ), _net_ip )::varchar(150)
          WHERE bd_minr = _minr
            AND bd_end IS NULL
          ;

      -- Sonderfall Gehen in Raucherpause (103)
      -- Präsenzzeit bleibt offen und Eintrag von Raucherpause in Abwesenheiten mit Zeitstempel wird erstellt.
      ELSIF   _abw_id = _id_raucherpause THEN
          -- Ende-Datum wird ggf. nach Kommen aus Pause korrigiert (Raucherpause über Mitternacht). Hier zur Konsistenz aktueller Zustand.
          INSERT INTO bdepab ( bdab_minr, bdab_anf,      bdab_anft,                bdab_end,      bdab_aus_id,      bdab_stu )
          VALUES             ( _minr,     _stempel_date, _stempel_zeitpunkt::time, _stempel_date, _id_raucherpause, 0 )
          ;

      END IF;

      -- Aufträge unterbrechen
      -- Nur durchführen, wenn die Abweichung der einzutragenden Stempelzeit +-5 Minuten zu jetzt ist (Stichwort: verzögerte Datenübertragung, falsche Uhrzeit).
      IF ( currenttime() - _stempel_zeitpunkt ) BETWEEN '-5 minutes'::interval AND '+5 minutes'::interval THEN
          -- Aufträge mit Abwesenheits-ID unterbrechen
          PERFORM
              tpersonal.bde__bdea__stempeln__ende__offene(
                  _minr,
                  _stempel_zeitpunkt,
                  -- Beim Gehen Standard über Terminal werden Aufträge mit Pause (110) unterbrochen.
                  -- Sehr fraglich, da diese auch wieder automatisch angestempelt werden (von individ. WT bis heute).
                  coalesce( _abw_id, _id_pause )
              )
          ;

      END IF;


      RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;
--

--
CREATE OR REPLACE FUNCTION TPersonal.bde__status__get(
      _minr           integer,                  -- Mitarbeiter-Nr.
      _cur_state_only boolean = false,          -- nur Zustand aktuell ermitteln, keine Summen (Tag, Konto)
                                                -- zur Entscheidung für Kommen oder Gehen im Status vorm Kommen/Gehen
      _target_time    timestamp = currenttime() -- Ziel-Zeitpunkt der Statusabfragen (insb. für CI und zukünftige Anforderungen)
  ) RETURNS TPersonal.bde__status AS $$
  DECLARE
      _bde__status                TPersonal.bde__status;  -- für RETURN

      _current_time               CONSTANT timestamp := currenttime();  -- aktueller Zeitpunkt der Abfrage
      _current_date               CONSTANT date := _target_time::date;  -- Ziel-Datum der Statusabfragen
      _current_wt                 CONSTANT date := coalesce( TPersonal.bdep__individwt__get_date__by_bd_anf_minr( _target_time, _minr )::date, _current_date );

      _id_mindestpause            CONSTANT integer := 101; -- feste ID der Mindestpause
      _id_raucherpause            CONSTANT integer := 103; -- feste ID der Raucherpause
      _id_pause                   CONSTANT integer := 110; -- feste ID der Pause

      _bdepab_rec                 record; -- Daten aus Abwesenheiten (Raucherpause u.a.)
      _bdep_rec                   record; -- Daten aus Präsenzzeiten (für Pauseneintrag)
  BEGIN
      -- zentrale Funtion zur Ermittlung aktueller BDE-Informationen


      -- Fehler: Ohne Angabe von Mitarbeiter-Nr., Option oder Zielzeitpunkt
      IF ( _minr IS NULL OR _cur_state_only IS NULL OR _target_time IS NULL ) THEN
          RAISE EXCEPTION '%', lang_text( 16834 );
      END IF;


      _bde__status.bs_minr  := _minr;
      _bde__status.bs_time  := _current_time;

      -- letzte An- oder Abstempelung
      -- Kontext ergibt sich aus bs_is_present
      _bde__status.bs_bd_end_anf_last :=
            coalesce( bd_end, bd_anf )
          FROM bdep
          WHERE bd_minr = _minr
          ORDER BY bd_anf DESC
          LIMIT 1
      ;
      _bde__status.bs_bd_end_anf_rund_last :=
            coalesce( bd_end_rund, bd_anf_rund )
          FROM bdep
          WHERE bd_minr = _minr
          ORDER BY bd_anf DESC
          LIMIT 1
      ;

      -- erste Anstempelung am individ. Werktag
      _bde__status.bs_bd_anf_rund_firstofday :=
            bd_anf_rund
          FROM bdep
          WHERE bd_minr = _minr
            AND bd_individwt_mpl_date = _current_wt
          ORDER BY bd_anf
          LIMIT 1
      ;


      -- Präsenz und Abwesenheiten:
          -- Sonderfall Raucherpause (103)
          -- Präsenzzeit bleibt offen und Eintrag von Raucherpause in Abwesenheiten mit Zeitstempel ist vorhanden.
          SELECT
            bdab_anf,
            bdab_anft
          FROM bdepab
          WHERE bdab_minr   = _minr
            AND bdab_aus_id = _id_raucherpause
            AND bdab_endt IS NULL
            -- ggf. Raucherpause über Mitternacht.
            AND bdab_anf BETWEEN _current_wt AND _current_date
          ORDER BY bdab_id DESC
          LIMIT 1
          INTO _bdepab_rec
          ;

          -- Raucherpause vorhanden
          IF _bdepab_rec IS NOT NULL THEN
              _bde__status.bs_is_present  := false;
              _bde__status.bs_abw_id      := _id_raucherpause;
              _bde__status.bs_abw_seit    := _bdepab_rec.bdab_anf + _bdepab_rec.bdab_anft; -- date + time = timestamp


          -- Keine Raucherpause, also Anwesenheit oder andere Abwesenheit
          ELSE
              -- MA ist anwesend.
              -- Auch wenn die Stempelung alt/unsinnig ist, gilt er als anwesend, siehe Fehlerbehandlung per bdep__a_20_iu__insert__bdep__fehlstempelung.
              _bde__status.bs_is_present := EXISTS(SELECT true FROM bdep WHERE bd_minr = _minr AND bd_end IS NULL); -- ggf. Widerspruch durch Unsinn mit insert_by = 'NT'


              -- Wenn MA nicht anwesend, dann spezifische Abwesenheit ermitteln.
              IF NOT _bde__status.bs_is_present THEN
                  -- Sonderfall Pause (110)
                    -- Präsenzzeit wird geschlossen mit Vermerk der Abwesenheit 110, aber ohne Eintrag in Abwesenheiten.
                  -- Zugriff auf letzten Präsenzzeit-Eintrag notwendig.
                  SELECT
                    bd_aus_id,
                    bd_end_rund
                  FROM bdep
                  WHERE bd_minr = _minr
                    AND bd_individwt_mpl_date = _current_wt
                  ORDER BY bd_anf DESC
                  LIMIT 1
                  INTO _bdep_rec
                  ;

                  -- Wenn dieser Pause (110) vermerkt hat, dann ist es eine Pause, sonst leer lassen und über Abwesenheiten gehen.
                  IF _bdep_rec.bd_aus_id = _id_pause THEN
                      _bde__status.bs_abw_id    := _id_pause;
                      _bde__status.bs_abw_seit  := _bdep_rec.bd_end_rund;


                  -- alle anderen ggf. eingetragenen Abwesenheiten
                  ELSE

                      SELECT
                        bdab_aus_id,
                        bdab_anf
                      FROM bdepab
                      WHERE bdab_minr = _minr
                        -- ohne Mindest- und Raucherpausen
                        AND bdab_aus_id NOT IN ( _id_mindestpause, _id_raucherpause )
                        AND _current_wt BETWEEN bdab_anf AND bdab_end
                      ORDER BY bdab_id DESC
                      LIMIT 1
                      INTO _bdepab_rec
                      ;

                      _bde__status.bs_abw_id    := _bdepab_rec.bdab_aus_id;
                      _bde__status.bs_abw_seit  := _bdepab_rec.bdab_anf;
                  END IF;

              END IF;

          END IF;
      --

      -- letzte MA der Firma
      _bde__status.bs_bd_minr_is_last_present :=
          -- Es gibt keinen anderen MA, der nicht abgestempelt ist,
          -- sonst gibt es einen MA, der noch angestempelt ist.
          NOT EXISTS(
            SELECT true
            FROM bdep
            WHERE bd_minr <> _minr
              AND bd_end IS NULL
          )
          -- und ich bin selbst anwesend (um letzter zu sein).
          AND _bde__status.bs_is_present
      ;


      -- auch Summen (Tag, Konto) ermitteln
      IF NOT _cur_state_only THEN

          -- Präsenzzeit IST am aktuellen individ. Werktag
          -- hier schon abzgl. bd_gleitpause und bd_blockpause, aber ohne Abzug von Pausen durch bdepab (z.B. Mindestpause).
          _bde__status.bs_pre_zeit_ist :=
                sum(
                    CASE
                      WHEN bd_end IS NOT NULL THEN
                          bd_saldo
                      ELSE
                          timediff( bd_anf, _current_time )
                    END
                )
              FROM bdep
              WHERE bd_minr = _minr
                AND (
                         bd_end IS NULL
                      OR bd_individwt_mpl_date = _current_wt
                )
          ;

          -- ohne Daten wenigstens 0
          _bde__status.bs_pre_zeit_ist := coalesce( _bde__status.bs_pre_zeit_ist, 0 );


          -- Abwesenheiten IST am aktuellen individ. Werktag
            -- mit Zeiten: Mindestpause, nachgetragene Pause, Schule usw.
            -- Urlaub und KranK haben keine bdab_stu
            -- Raucherpause hat bdab_stu = 0
          _bde__status.bs_abw_zeit_ist :=
                sum( bdab_stu )
              FROM bdepab
              WHERE bdab_minr = _minr
                AND _current_wt BETWEEN bdab_anf AND bdab_end
                AND bdab_stu IS NOT NULL
          ;

          -- Das Ergebnis wird hier negiert entspr. des Feldnamens: Abwesenheitszeit muss positiv sein.
            -- Abwesenheiten sind auf DB negativ und Gutschriften/Korrekturen positiv gespeichert.
            -- ohne Daten wenigstens 0
          _bde__status.bs_abw_zeit_ist := - coalesce( _bde__status.bs_abw_zeit_ist, 0 );


          -- Raucherpausen IST am aktuellen individ. Werktag
            -- ohne Daten wenigstens 0 mittels Funktion
          _bde__status.bs_raucherpausen_ist := TPersonal.bdep__raucherpausen_summe__get( _minr, _current_wt );


          -- tatsächliche Arbeitszeit IST am aktuellen individ. Werktag
            -- Arbeitszeit entspr. der gesetzlichen Definition
            -- Präsenzzeit abzgl. Abwesenheiten und abzgl. Raucherpausen
            -- Nur bei vorhandener Präsenzzeit. Anderer Anwendungsfall unklar.
          IF _bde__status.bs_pre_zeit_ist > 0 THEN

              _bde__status.bs_arbeitszeit_ist :=
                  greatest(
                        _bde__status.bs_pre_zeit_ist
                      - _bde__status.bs_abw_zeit_ist
                      - _bde__status.bs_raucherpausen_ist
                      -- mindestens 0
                      , 0
                  )
              ;

          END IF;

          -- ohne Daten wenigstens 0
          _bde__status.bs_arbeitszeit_ist := coalesce( _bde__status.bs_arbeitszeit_ist, 0 );


          -- aktuelles Stundekonto
            -- ohne Daten wenigstens 0
          _bde__status.bs_konto_stunden := coalesce( TPersonal.llv__mitpln__ll_stuko__stundenkonto__calc( _minr, _current_date, false ), 0 );


          -- aktueller Resturlaub
            -- mit "- get_urlaubstage_geplant(_minr, true)" ist es der noch planbare Urlaub
            -- ohne Daten wenigstens 0
          _bde__status.bs_konto_urlaub  := coalesce( tpersonal.llv__urlaub__act( _minr, true ), 0 );


          -- Stunden der unverbuchten Raucherpausen bis Ziel-Datum der Statusabfrage ermitteln

          _bde__status.bs_konto_raucherpausen :=
                  tpersonal.llv__bdpab__rpause__round(
                      -- für Mitarbeiter
                      _minr,
                      -- INDEX bdepab_bdab_buch
                      '-Infinity',
                      -- Ziel-Datum der Statusabfrage
                      _current_date,
                      -- nur unverbuchte
                      true
                  )
              ;

          -- ohne Daten wenigstens 0
          _bde__status.bs_konto_raucherpausen := coalesce( _bde__status.bs_konto_raucherpausen, 0 );


      END IF;


      RETURN _bde__status;
  END $$ LANGUAGE plpgsql STABLE;
--

--
CREATE OR REPLACE FUNCTION TPersonal.llv__ll_rfid__to__ll_minr(
      _rfid         varchar,                    -- RFID
      _target_time  timestamp = currenttime(),  -- Ziel-Zeitpunkt der Prüfung
                                                  -- Nachtragen von Stempelungen muss RFID-Gültigkeit per Stempelzeitpunkt vs. Datum des Ausscheidens prüfen.
                                                  -- CI-Tests per konkretem Zeitpunkt unabhängig current_date.

      OUT minr      integer,  -- Mitarbeiter-Nr.
      OUT msg_error varchar   -- Fehler-Text zur Ausgabe
  ) RETURNS record AS $$
  DECLARE
      _current_date     CONSTANT date := _target_time::date;  -- Abfragen auf konstantes Datum schneller

      _valid_minr       integer[];  -- gültige (nicht ausgeschiedene) Mitarbeiter mit dieser RFID
      _invalid_minr     integer[];  -- ungültige (ausgeschiedene) Mitarbeiter mit dieser RFID
      _valid_minr_qty   integer;    -- abgeleitete Anzahl
      _invalid_minr_qty integer;
  BEGIN
      -- Funktion zur Validierung und Ermittlung der Mitarbeiter-Nr. zur RFID


      -- Fehler: RFID nicht angegeben
      IF _rfid IS NULL THEN
          msg_error := lang_text( 16805 );

          RETURN;
      END IF;

      -- Fehler: RFID-Prüfung ohne Zielzeit nicht möglich
      IF _target_time IS NULL THEN
          msg_error := lang_text( 16824 );

          RETURN;
      END IF;


      -- führende Nullen entfernen
      -- Eintrag in Prodat muss ohne fixe Länge (ohne führende Nullen) sein.
      _rfid := ltrim( _rfid, '0' );


      -- Mitarbeiter (gültige, ausgeschiedene), denen die RFID zugewiesen ist.
      SELECT
        array_agg( ll_minr ORDER BY ll_minr ) FILTER ( WHERE coalesce( ll_endd, _current_date + 1 ) >  _current_date ) AS   valid_minr,
        array_agg( ll_minr ORDER BY ll_minr ) FILTER ( WHERE coalesce( ll_endd, _current_date + 1 ) <= _current_date ) AS invalid_minr
      FROM llv
      WHERE TSystem.ENUM_GetValue( ll_rfid, _rfid )
      INTO
        _valid_minr,
        _invalid_minr
      ;

      _valid_minr_qty   := coalesce( array_length(   _valid_minr, 1 ), 0 );
      _invalid_minr_qty := coalesce( array_length( _invalid_minr, 1 ), 0 );

      -- RFID ist an gültigen MA vergeben
      -- Ermittlung der zugehörigen Mitarbeiter-Nummer kann erfolgen
      IF    _valid_minr_qty = 1 THEN
          minr := _valid_minr[1];

      -- Fehler: RFID nicht vergeben
      ELSIF ( _valid_minr_qty + _invalid_minr_qty ) = 0 THEN
          msg_error := lang_text( 16362) ;

      -- Fehler: Mitarbeiter ausgeschieden: 1, 2, 3
      ELSIF _invalid_minr_qty > 0 THEN
          msg_error := format( lang_text( 16361 ), array_to_string( _invalid_minr, ', ' ) );

      -- Fehler: RFID mehrfach vergeben: 1, 2, 3
      ELSE -- _valid_minr_qty > 0
          msg_error := format( lang_text( 16363 ), array_to_string(   _valid_minr, ', ' ) );

      END IF;


      RETURN;
  END $$ LANGUAGE plpgsql STABLE;
--


--------------------------------------- zentrale BDE-Funktionen Ende ---------------------------------------

------------------------------------- zentrale Terminal-Funktionen Anfang ----------------------------------

--
CREATE OR REPLACE FUNCTION TPersonal.bde__stempeln__terminal__execute(
      -- spez. Terminal
      _manufacturer       varchar,  -- Hersteller
      _device_type        varchar,  -- Gerät
      _software_version   varchar,  -- Programm-Version auf dem Gerät

      -- Stempelprozess-Daten
      _rfid               varchar,                    -- RFID
      _minr               integer,                    -- Mitarbeiter-Nr.
      _stempel_zeitpunkt  timestamp = currenttime(),  -- Stempelzeit (Kommen oder Gehen)
      _abw_id             integer = null,             -- Abwesenheit (Gehen mit Abwesenheit)
      _terminal_info      varchar = null,             -- Terminal-Infos (Geräte-Nr, SN usw.)
      _net_ip             varchar = null,             -- IP des Terminals
      _bde__status_only   boolean = false,            -- nur BDE-Infos abfragen

      -- Ausgaben für Terminal
      OUT event_id        integer,  -- Ereignis-ID für ausgeführten Prozess (binärcodier)
      OUT msg_error       varchar,  -- Fehler
      OUT msg_warning     varchar,  -- Warnungen
      OUT msg_info        varchar   -- menschenlesbare Informationen zum Stempelprozess
  ) RETURNS record AS $$
  DECLARE
      _bde_terminal_return_rec record;
  BEGIN
      -- zentrale Terminal-Funktion für alle Terminal-Typen
        -- steuert anhand Eingabe spezifische Terminal-Funktionen an. Darin werden spezifische Zustände entspr. Terminal-Typ behandelt.
        -- siehe:
          -- bde__stempeln__terminal__prodat__execute - für Prodat-Stemploberfläche
          -- bde__stempeln__terminal__datafox_evo__execute - für Datafox-Terminal


      -- Record für RETURN initialisieren
      SELECT
        event_id,
        msg_error,
        msg_warning,
        msg_info
      INTO
        _bde_terminal_return_rec
      ;

      _manufacturer := upper( _manufacturer );

      -- CIMPCS
      IF    _manufacturer = 'CIMPCS' THEN -- exemplarisch
          -- Prodat-Zeiterfassung
          IF _device_type = 'TFormBDEGetTime' THEN -- exemplarisch
              -- Stempelprozess mit Rückgaben durchführen
              _bde_terminal_return_rec :=
                  TPersonal.bde__stempeln__terminal__prodat__execute(
                        _device_type,
                        _software_version,
                        _rfid,
                        _minr,
                        _stempel_zeitpunkt,
                        _abw_id,
                        _terminal_info,
                        _net_ip,
                        _bde__status_only
                  )
              ;

          -- Fehler: Gerät "%s" nicht unterstützt
          ELSE
              _bde_terminal_return_rec.msg_error := format( lang_text( 16804 ), coalesce( _device_type, 'null' ) );
          END IF;

      -- DATAFOX
      ELSIF _manufacturer = 'DATAFOX' THEN
          -- EVO-Geräte
          IF _device_type IN ( 'evo28', 'evo35', 'evo35u', 'evo43' ) THEN
              -- Stempelprozess mit Rückgaben durchführen
              _bde_terminal_return_rec :=
                  TPersonal.bde__stempeln__terminal__datafox_evo__execute(
                        _device_type,
                        _software_version,
                        _rfid,
                        _stempel_zeitpunkt,
                        _abw_id,
                        _terminal_info,
                        _net_ip,
                        _bde__status_only
                  )
              ;

          -- Fehler: Gerät "%s" nicht unterstützt
          ELSE
              _bde_terminal_return_rec.msg_error := format( lang_text( 16804 ), coalesce( _device_type, 'null' ) );
          END IF;

      -- NEXT
      -- ELSIF _manufacturer = 'NEXT' THEN

      -- Fehler: Hersteller "%s" und Gerät "%s" nicht unterstützt.
      ELSE
          _bde_terminal_return_rec.msg_error := format( lang_text( 16832 ), coalesce( _manufacturer, 'null' ), coalesce( _device_type, 'null' ) );
      END IF;

      event_id    := _bde_terminal_return_rec.event_id;
      msg_error   := _bde_terminal_return_rec.msg_error;
      msg_warning := _bde_terminal_return_rec.msg_warning;
      msg_info    := _bde_terminal_return_rec.msg_info;


      RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;
--

--
CREATE OR REPLACE FUNCTION TPersonal.bde__stempeln__terminal__prodat__execute(
      _device_type        varchar,                    -- Prodat-Modul
      _software_version   varchar,                    -- Prodat-Version
      _rfid               varchar,                    -- RFID
      _minr               integer,                    -- Mitarbeiter-Nr.
      _stempel_zeitpunkt  timestamp = currenttime(),  -- Stempelzeit (Kommen oder Gehen)
      _abw_id             integer = null,             -- Abwesenheit (Gehen mit Abwesenheit)
      _terminal_info      varchar = null,             -- Terminal-Infos (Geräte-Nr, SN usw.)
      _net_ip             varchar = null,             -- IP des Terminals
      _bde__status_only   boolean = false,            -- nur BDE-Infos abfragen

      -- Ausgaben für Terminal
      OUT event_id        integer,  -- Ereignis-ID für ausgeführten Prozess (binärcodier)
      OUT msg_error       varchar,  -- Fehler direkt per EXCEPTION
      OUT msg_warning     varchar,  -- Warnungen
      OUT msg_info        varchar   -- menschenlesbare Informationen zum Stempelprozess
  ) RETURNS record AS $$
  DECLARE
      _bde__stempeln_rec    record;
      _bde__status_before   TPersonal.bde__status;  -- Status vorm Kommen/Gehen
      _bde__status_after    TPersonal.bde__status;  -- Status nach Kommen/Gehen
  BEGIN
      -- exemplarisch, noch TODO
      -- zentrale Funktion für Prodat-Terminal (Zeiterfassung TFormBDEGetTime)
        -- Terminal gibt einfach "Aktion" an diese Funktion, ohne den Status (angestempelt bzw. abgestempelt) zu kennen.
        -- Die Funktion ermittelt den Status und führt die korrekte Aktion aus. Dem Terminal wird das mitgeteilt und dem Bediener angezeigt.


      -- Fehler: Ohne Angabe von Mitarbeiter-Nr. oder RFID
      IF ( _rfid IS NULL AND _minr IS NULL ) THEN
          RAISE EXCEPTION '%', lang_text( 16835 );
      END IF;

      -- Fehler: ohne Angabe von Stempelzeit
      IF NOT _bde__status_only AND _stempel_zeitpunkt IS NULL THEN
          RAISE EXCEPTION '%', lang_text( 16806 );
      END IF;


      -- Mitarbeiter-Nr. über RFID ermitteln
      IF _minr IS NULL AND _rfid IS NOT NULL THEN
          -- Mitarbeiter-Nr. zur RFID ermitteln.
          SELECT
            llv__ll_rfid__to__ll_minr.minr,
            llv__ll_rfid__to__ll_minr.msg_error
          FROM TPersonal.llv__ll_rfid__to__ll_minr( _rfid, _stempel_zeitpunkt )
          INTO
            _minr,
            msg_error
          ;

          -- Fehler bei der Ermittlung der Mitarbeiter-Nr. zur RFID ausgeben
          IF msg_error IS NOT NULL THEN
              RAISE EXCEPTION '%', msg_error;
          END IF;
      END IF;


      -- nur BDE-Infos abfragen
      IF _bde__status_only THEN
          _bde__status_after := TPersonal.bde__status__get( _minr );

      -- Stempeln durchführen
      -- und BDE-Status-Infos vor und nach der Aktion für Ausgabe speichern
      ELSE
          -- Stempeln inkl. BDE-Infos
          SELECT
            bde__stempeln__execute.bde__status_before,
            bde__stempeln__execute.bde__status_after
          FROM  TPersonal.bde__stempeln__execute(
                    _minr,
                    _stempel_zeitpunkt,
                    _abw_id,
                    _terminal_info,
                    _net_ip
                )
          INTO
            _bde__stempeln_rec
          ;

          -- geht nicht sofort in die vars, weil: "ERROR:  record or row variable cannot be part of multiple-item INTO list"
          _bde__status_before := _bde__stempeln_rec.bde__status_before;
          _bde__status_after  := _bde__stempeln_rec.bde__status_after;

      END IF;

      -- Ereignis-ID aufgrund Stempelprozess ermitteln.
      event_id :=
          TPersonal.bde__stempeln__terminal_event_id__get(
              _bde__status_before.bs_is_present,
              _bde__status_after.bs_is_present,
              _bde__status_only
          )
      ;

      -- hier dann Status in menschenlesbare Informationen umwandeln, vgl. bde__stempeln__terminal__datafox_evo__execute
      msg_info :=
          lang_text( 28801 ) || _bde__status_before::varchar || E'\n' ||
          lang_text( 28800 ) || _bde__status_after::varchar
      ;


      RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;
--

--
CREATE OR REPLACE FUNCTION TPersonal.bde__stempeln__terminal__datafox_evo__execute(
      _device_type        varchar,                    -- Geräteversion vom Terminal
      _software_version   varchar,                    -- Programm-Version (ggf. für Zukunft)
      _rfid               varchar,                    -- RFID
      _stempel_zeitpunkt  timestamp = currenttime(),  -- Stempelzeit (Kommen oder Gehen)
      _abw_id             integer = null,             -- Abwesenheit (Gehen mit Abwesenheit)
      _terminal_info      varchar = null,             -- Terminal-Infos (Geräte-Nr, SN usw.)
      _net_ip             varchar = null,             -- IP des Terminals
      _bde__status_only   boolean = false,            -- nur BDE-Infos abfragen

      -- Ausgaben für Terminal
      OUT event_id        integer,  -- Ereignis-ID für ausgeführten Prozess
      OUT msg_error       varchar,  -- Fehler
      OUT msg_warning     varchar,  -- Warnungen
      OUT msg_info        varchar   -- menschenlesbare Informationen zum Stempelprozess:
                                        -- Herbert Schmöde
                                        -- Status: Anwesend
                                        -- Arbeitszeit: 04:24 h
                                        -- Raucherpause: 00:04 h
                                        -- Stundenkonto: +14:30 h
                                        -- Urlaubstage: 21 Tage
  ) RETURNS record AS $$
  DECLARE
      _minr                 integer;                -- Mitarbeiter-Nr.
      _bde__stempeln_rec    record;
      _bde__status_before   TPersonal.bde__status;  -- Status vorm Kommen/Gehen
      _bde__status_after    TPersonal.bde__status;  -- Status nach Kommen/Gehen

      _message_templates    record;
      _additional_msg_data  jsonb;
      _text_values_pretty   jsonb;
  BEGIN
      -- zentrale Funktion für Terminal Datafox EVO
        -- Versionen: evo28, evo35, evo35u, evo43


      -- Fehler: ohne Angabe von Stempelzeit
      IF NOT _bde__status_only AND _stempel_zeitpunkt IS NULL THEN
          msg_error := lang_text( 16806 );

          RETURN;
      END IF;


      -- Mitarbeiter-Nr. zur RFID ermitteln.
      SELECT
        llv__ll_rfid__to__ll_minr.minr,
        llv__ll_rfid__to__ll_minr.msg_error
      FROM TPersonal.llv__ll_rfid__to__ll_minr( _rfid, _stempel_zeitpunkt )
      INTO
        _minr,
        msg_error
      ;

      -- Fehler bei der Ermittlung der Mitarbeiter-Nr. zur RFID ausgeben
      IF msg_error IS NOT NULL THEN
          RETURN;
      END IF;


      -- nur BDE-Infos abfragen
      IF _bde__status_only THEN
          _bde__status_after := TPersonal.bde__status__get( _minr );

      -- Stempeln durchführen
      -- und BDE-Status-Infos vor und nach der Aktion für Ausgabe speichern
      ELSE
          -- Daten vorm Stempeln zur Fehlerbehandlung ermitteln
          _bde__status_before := TPersonal.bde__status__get( _minr, true );

          -- Fehler: Nur 1 Stempelung pro 10 Sekunden erlaubt
          -- Anpassung 18733 bzw 18727 von vorher Prüfung gerundeter Zeiten auf neu ungerundete Zeiten
          IF    _bde__status_before.bs_bd_end_anf_last BETWEEN _stempel_zeitpunkt - interval '10 seconds' AND _stempel_zeitpunkt THEN
              msg_error := lang_text( 16807 );

              RETURN;

          -- Warnung: Doppelte Stempelung innerhalb von 5 Minuten
          ELSIF _bde__status_before.bs_bd_end_anf_last BETWEEN _stempel_zeitpunkt - interval '5 minute' AND _stempel_zeitpunkt THEN
              msg_warning := lang_text( 16808 );

          END IF;

          -- Zurücksetzen für Stempel-Prozess
          _bde__status_before := null;

          -- Stempeln inkl. BDE-Infos
          SELECT
            bde__stempeln__execute.bde__status_before,
            bde__stempeln__execute.bde__status_after
          FROM  TPersonal.bde__stempeln__execute(
                    _minr,
                    _stempel_zeitpunkt,
                    _abw_id,
                    _terminal_info,
                    _net_ip
                )
          INTO
            _bde__stempeln_rec
          ;

          -- geht nicht sofort in die vars, weil: "ERROR:  record or row variable cannot be part of multiple-item INTO list"
          _bde__status_before := _bde__stempeln_rec.bde__status_before;
          _bde__status_after  := _bde__stempeln_rec.bde__status_after;

      END IF;

      -- Ereignis-ID aufgrund Stempelprozess ermitteln.
      event_id :=
          TPersonal.bde__stempeln__terminal_event_id__get(
              _bde__status_before.bs_is_present,
              _bde__status_after.bs_is_present,
              _bde__status_only
          )
      ;

      --  BDE-Prozess bzw. -Status als Terminal-Texte (Info, Warnung, Fehler) ermitteln.
          -- Text-Templates mit Text-Variablen je nach Prozess bzw. Status holen.
          _message_templates :=
              TPersonal.bde__stempeln__terminal__datafox_evo__message_templates__fetch(
                  _device_type,
                  _software_version,
                  event_id,
                  _bde__status_before,
                  _bde__status_after
              )
          ;

          -- Zusätzliche Werte (abseits vom BDE-Status) fürs Formatieren schön verpacken.
          _additional_msg_data :=
              jsonb_build_object(
                  '_rfid', _rfid,
                  '_stempel_zeitpunkt', _stempel_zeitpunkt
              )
          ;

          -- Werte für Texte formatieren.
          _text_values_pretty :=
              TPersonal.bde__stempeln__terminal__message__format__fetch(
                  _bde__status_after,
                  _additional_msg_data
              )
          ;

          -- Template für Ereignis-Text hinzufügen
          _text_values_pretty := _text_values_pretty || jsonb_build_object( '_clock_event', _message_templates.template_clock_event );

          -- Info mit formatierten Werten füllen
          msg_info :=
              TPersonal.bde__stempeln__terminal__message__textvar__replace(
                  _message_templates.template_info,
                  _text_values_pretty,
                  -- Caption für evo43 in erste Zeile
                  _event_caption_first_line => _device_type = 'evo43'
              )
          ;

          -- Warnung
          msg_warning := coalesce( _message_templates.template_warning, msg_warning );

          -- Fehler
          msg_error   := coalesce( _message_templates.template_error, msg_error );
      --


      RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;
--

--
CREATE OR REPLACE FUNCTION TPersonal.bde__stempeln__terminal__datafox_evo__message_templates__fetch(
      _device_type        varchar,                -- Geräteversion vom Terminal (nur vorgesehen, aktuell nicht benötigt)
      _software_version   varchar,                -- Programm-Version (nur vorgesehen, aktuell nicht benötigt)
      _terminal_event_id  integer,                -- Ereignis-ID des ausgeführten Prozesses
      _bde__status_before TPersonal.bde__status,  -- Status vorm Kommen/Gehen
      _bde__status_after  TPersonal.bde__status,  -- Status nach Kommen/Gehen

      OUT template_error        varchar,  -- Fehler (nur vorgesehen, aktuell nicht benötigt)
      OUT template_warning      varchar,  -- Warnungen
      OUT template_info         varchar,  -- Informationen zu Stempelprozess bzw. Statusabfrage
      OUT template_clock_event  varchar   -- Ereignis des Stempelprozesses
  ) RETURNS record AS $$
  DECLARE
      _id_raucherpause  CONSTANT integer := 103; -- feste ID der Raucherpause
      _id_pause         CONSTANT integer := 110; -- feste ID der Pause
  BEGIN
      -- Funktion zur Ermittlung der Vorlagen-Texte (inkl. Text-Variablen) für Terminalausgaben je nach Prozess und Abfrage


      -- Ereigniss-Texte
      -- Kommen
      IF      _terminal_event_id = 1 THEN
          template_clock_event := lang_text( 16813 );

      -- Gehen
      ELSIF   _terminal_event_id = 2 THEN
          template_clock_event := lang_text( 16817 );

      -- Info
      ELSIF   _terminal_event_id = 4 THEN
          template_clock_event := lang_text( 16809 );

      -- Undefiniertes Ereignis
      ELSE
          template_clock_event := lang_text( 16822 );

      END IF;


      -- Terminalausgaben
      -- Kommen
      IF    _terminal_event_id = 1 THEN
          -- Standard
          IF      NOT _bde__status_before.bs_is_present THEN
              -- Kommen aus Pause
              IF _bde__status_before.bs_abw_id IN ( _id_raucherpause, _id_pause ) THEN
                  template_info := lang_text( 16815 );  -- Template für Kommen aus Pause

              -- normal Kommen (auch aus anderen Abwesenheiten)
              ELSE
                  template_info := lang_text( 16814 );  -- Template für Kommen

              END IF;

          END IF;

          -- Sonderfall: Kommen aufgrund langer Fehlstempelungen (>15 h)
          IF      _bde__status_before.bs_is_present
              AND _bde__status_after.bs_is_present
          THEN
              template_info     := lang_text( 16816 );  -- Template für Kommen (Abstempeln vergessen)
              template_warning  := lang_text( 16836 );  -- Warnung: Abstempeln vergessen

          END IF;


      -- Gehen
      ELSIF _terminal_event_id = 2 THEN
          -- Gehen in Pause
          IF _bde__status_after.bs_abw_id IN ( _id_raucherpause, _id_pause ) THEN
              template_info := lang_text( 16819 );  -- Template für Gehen in Pause

          -- Gehen mit anderer Abwesenheit
          ELSIF _bde__status_after.bs_abw_id IS NOT NULL THEN
              template_info := lang_text( 16820 );  -- Template für Gehen mit Abwesenheit

          -- Gehen und letzter Mitarbeiter
          ELSIF _bde__status_before.bs_bd_minr_is_last_present THEN
              template_info := lang_text( 16821 );  -- Template für Gehen mit Letzter Mitarbeiter
          -- normal Gehen
          ELSE
              template_info := lang_text( 16818 );  -- Template für Gehen

          END IF;


      -- Abfrage von BDE-Infos
      ELSIF _terminal_event_id = 4 THEN
          -- Templates je Status

          -- Anwesend
          IF _bde__status_after.bs_is_present THEN
              template_info := lang_text( 16810 ); -- Template für BDE-Infos mit Anwesend seit: 09:00 (05:59 h)

          -- Abwesend mit Pause
          ELSIF _bde__status_after.bs_abw_id IN ( _id_raucherpause, _id_pause ) THEN
              template_info := lang_text( 16811 ); -- Template für BDE-Infos mit Pause seit: 12:00

          -- Abwesend
          ELSE
              template_info := lang_text( 16812 ); -- Template für BDE-Infos mit Abwesend

          END IF;


      -- Undefinierter Vorgang
      ELSE
          template_info     := lang_text( 16823 ); -- Template für undefinierten Vorgang
          template_warning  := lang_text( 16837 ); -- Warnung: Undefinierter Vorgang
      END IF;


      RETURN;
  END $$ LANGUAGE plpgsql STABLE;
--

--
CREATE OR REPLACE FUNCTION TPersonal.bde__stempeln__terminal_event_id__get(
      _is_present_before  boolean,  -- Präsenz-Status vorm Kommen/Gehen
      _is_present_after   boolean,  -- Präsenz-Status nach Kommen/Gehen
      _bde_info           boolean   -- BDE-Infos abfragen
  ) RETURNS integer AS $$
  DECLARE
      _event_id   integer := 0;
  BEGIN
      -- Funktion zur Ermittlung der Ereignis-ID des ausgeführten Prozesses
        -- binärcodier für mgl. Kombinationen
        -- Kommen = 1, Gehen = 2, Info = 4, nächster Fall wäre = 8
        -- also: Kommen + Info = 5, Gehen + Info = 6
        -- 0 bedeutet damit unbekannt
        -- Beachte NULL-Zustände


      -- 1: Kommen
      IF  -- normal
          (
                  NOT _is_present_before
              AND _is_present_after
          )

          -- Sonderfall: Kommen aufgrund langer Fehlstempelungen (>15 h)
          -- explizit ausgewiesen für Eindeutigkeit und für zukünftige Bearbeitung dieses Falls
          OR  (
                  _is_present_before
              AND _is_present_after
          )
      THEN
          _event_id := _event_id | 1; -- bitwise OR
      END IF;

      -- 2: Gehen
      IF      _is_present_before
          AND NOT _is_present_after
      THEN
          _event_id := _event_id | 2;
      END IF;

      -- 4: BDE-Infos
      IF _bde_info THEN
          _event_id := _event_id | 4;
      END IF;


      RETURN _event_id;
  END $$ LANGUAGE plpgsql IMMUTABLE;
--

--
CREATE OR REPLACE FUNCTION TPersonal.bde__stempeln__terminal__message__format__fetch(
      _bde__status      TPersonal.bde__status,  -- abgefragter Status
      _additional_data  jsonb                   -- zusätzliche Daten
  ) RETURNS jsonb AS $$
      -- zentrale Funktion zur Formatierung von BDE-Infos für Terminal-Texte


      SELECT
        to_jsonb( pretty )
      FROM (
          SELECT
            -- vollständiger Name des Mitarbeiters
            nameaufloesen( _bde__status.bs_minr )
            AS _employee_full_name,

            -- erste Anstempelung am individ. Werktag
            to_char( _bde__status.bs_bd_anf_rund_firstofday, 'HH24:MI' )
            AS _first_clock_today,

            -- Präzenzzeit-Ist
            to_char( (_bde__status.bs_pre_zeit_ist        || ' h')::interval, 'HH24:MI' ) || ' h'
            AS _attendance_time_span,

            -- Abwesenheitszeit-Ist
            to_char( (_bde__status.bs_abw_zeit_ist        || ' h')::interval, 'HH24:MI' ) || ' h'
            AS _absence_time_span,

            -- Raucherpause-Ist
            to_char( (_bde__status.bs_raucherpausen_ist   || ' h')::interval, 'HH24:MI' ) || ' h'
            AS _cig_break_time_span,

            -- Arbeitszeit-Ist
            to_char( (_bde__status.bs_arbeitszeit_ist     || ' h')::interval, 'HH24:MI' ) || ' h'
            AS _working_time_span,

            -- unverbuchte Raucherpausen-Konto
            to_char( (_bde__status.bs_konto_raucherpausen || ' h')::interval, 'HH24:MI' ) || ' h'
            AS _cig_break_account,

            -- Abwesenheits-ID
            _bde__status.bs_abw_id::varchar
            AS _absence_id,

            -- letzter Eintrag zur Abwesenheit (Datum und Urzeit bei Pause/Raucherpause)
            to_char( _bde__status.bs_abw_seit, 'HH24:MI' )
            AS _absent_since,

            -- Stundenkonto
            -- positiv oder negativ darstellen, negative Werte sehen sonst so aus: -01:-23
              -- erst Vorzeichen extrahieren
                  to_char( _bde__status.bs_konto_stunden, 'SG' )
              -- dann per Absolutwert darstellen
              ||  to_char( (abs( _bde__status.bs_konto_stunden ) || ' h')::interval, 'HH24:MI' )
              -- in Stunden
              ||  ' h'
            AS _time_account,

            -- Urlaubskonto
            -- trim: Dezimalzeichen am Ende bei ganzen Zahlen entfernen
            trim( to_char( _bde__status.bs_konto_urlaub, 'FM9990D99' ), trim( to_char( 0, 'D' ) ) )
            AS _leave_account,

            -- letzte Stempelung
            to_char( _bde__status.bs_bd_end_anf_rund_last, 'DD. TMMon HH24:MI' )
            AS _last_clock,


            -- zusätzliche Daten
            -- RFID ohne führende 0en
            ltrim( (_additional_data->>'_rfid')::varchar, '0' )
            AS _rfid,

            -- Datum Stempelzeit
            to_char( (_additional_data->>'_stempel_zeitpunkt')::timestamp, 'DD. TMMon IY' )
            AS _clock_date,

            -- Uhrzeit Stempelzeit
            to_char( (_additional_data->>'_stempel_zeitpunkt')::timestamp, 'HH24:MI' )
            AS _clock_time

      ) AS pretty
  $$ LANGUAGE sql STABLE;
--

--
CREATE OR REPLACE FUNCTION TPersonal.bde__stempeln__terminal__message__textvar__replace(
      _message                  varchar,
      _text_values              jsonb,
      -- Optionen
      _event_caption_first_line boolean = false -- Option für Datafox EVO43
  ) RETURNS varchar AS $$
  BEGIN
      -- zentrale Funktion zur Ersetzung der definierten Text-Variablen mit formatierten Werten.


      _message := replace( _message, '$employeeFullName$'    , coalesce( _text_values->>'_employee_full_name'   , '' ) );
      _message := replace( _message, '$rfid$'                , coalesce( _text_values->>'_rfid'                 , '' ) );

      _message := replace( _message, '$clockDate$'           , coalesce( _text_values->>'_clock_date'           , '' ) );
      _message := replace( _message, '$clockTime$'           , coalesce( _text_values->>'_clock_time'           , '' ) );
      _message := replace( _message, '$firstClockToday$'     , coalesce( _text_values->>'_first_clock_today'    , '' ) );
      _message := replace( _message, '$attendanceTimeSpan$'  , coalesce( _text_values->>'_attendance_time_span' , '' ) );
      _message := replace( _message, '$absenceTimeSpan$'     , coalesce( _text_values->>'_absence_time_span'    , '' ) );
      _message := replace( _message, '$cigBreakTimeSpan$'    , coalesce( _text_values->>'_cig_break_time_span'  , '' ) );
      _message := replace( _message, '$workingTimeSpan$'     , coalesce( _text_values->>'_working_time_span'    , '' ) );

      _message := replace( _message, '$absenceID$'           , coalesce( _text_values->>'_absence_id'           , '' ) );
      _message := replace( _message, '$absentSince$'         , coalesce( _text_values->>'_absent_since'         , '' ) );

      _message := replace( _message, '$timeAccount$'         , coalesce( _text_values->>'_time_account'         , '' ) );
      _message := replace( _message, '$leaveAccount$'        , coalesce( _text_values->>'_leave_account'        , '' ) );
      _message := replace( _message, '$cigBreakAccount$'     , coalesce( _text_values->>'_cig_break_account'    , '' ) );

      _message := replace( _message, '$lastClock$'           , coalesce( _text_values->>'_last_clock'           , '' ) );

      -- optionale Formatierung
      IF _event_caption_first_line THEN
          -- Caption der Textbox (unten links) steht in 1. Zeile
          _message := coalesce( _text_values->>'_clock_event', '' ) || E'\n' || _message;
      END IF;


      RETURN _message;
  END $$ LANGUAGE plpgsql IMMUTABLE;
--


------------------------------------- zentrale Terminal-Funktionen Ende ------------------------------------

-- Prüfen ob Stempelung Schichtbeginn markiert
CREATE OR REPLACE FUNCTION  Tpersonal.bdep__individwt__isfirstofday(
      in_bdanf  TIMESTAMP,
      in_bdminr INTEGER
  ) RETURNS BOOLEAN AS $$
  DECLARE
      flag      BOOLEAN := false;
      stamp_gap INTERVAL;
  BEGIN
    -- Abstand abgefragter Zeitpunkt zur vorrigen Stempelung
    stamp_gap :=
        in_bdanf - -- minus
        -- mein Vorgänger
        ( SELECT
            -- bei noch offener Stempelung bis jetzt bzw. maximal möglicher Stempelzeit, siehe bdep__b_20_iu
            -- Todo: Setting gesetzlicheRuhezeit existiert noch nicht
            max( COALESCE(bd_end, LEAST(currenttime(), bd_anf + (TSystem.Settings__GetInteger('gesetzlicheRuhezeit', 15) || ' hours')::INTERVAL )) ) AS previous_bd_end
          FROM bdep
          WHERE bd_minr = in_bdminr
            -- Performance
            AND timestamp_to_date(bd_anf) BETWEEN in_bdanf::DATE - 1 AND in_bdanf::DATE
            AND bd_anf < in_bdanf
        )
    ;

    -- Wenn null, dann maximaler Abstand
    stamp_gap := COALESCE(stamp_gap, INTERVAL '24 hours');

    -- eigentliche Ermittlung ob Schichtbeginn:
    -- Wenn mehr als 6h zwischen der aktuellen Zeit und der vorherigen Stempelzeit liegen -> definitiv Schichtbeginn
    IF stamp_gap > INTERVAL '6 hours' THEN
        RETURN true;

    -- Wenn keine 6 h zwischen der aktuellen und der vorherigen Stempelung liegen -> prüfen ob wenigstens > 5 (sonst könnte es auch eine Pause sein)
    -- zusätzliche Prüfung auf den llv_autoplan zum Tagesplan und dem Aktuellen Datum. Wenn vorhanden - dann auch Schichtbeginn
    ELSIF stamp_gap > INTERVAL '5 hours' THEN
        SELECT
          in_bdanf::TIME BETWEEN llpl_anf AND llpl_end
        INTO flag
        FROM mitpln
          LEFT JOIN llv_autoplan ON llpl_minr = mpl_minr
        WHERE mpl_minr = in_bdminr
          AND
            CASE
              WHEN EXTRACT( dow FROM in_bdanf::DATE ) BETWEEN 1 AND 4 THEN
                  llpl_tpl_name = mpl_tpl_name
              WHEN EXTRACT( dow FROM in_bdanf::DATE ) = 5 THEN
                  llpl_tpl_name_fr = mpl_tpl_name
              WHEN EXTRACT( dow FROM in_bdanf::DATE ) = 6 THEN
                  llpl_tpl_name_sa = mpl_tpl_name
              ELSE
                  llpl_tpl_name_so = mpl_tpl_name
            END
          AND mpl_date = in_bdanf::DATE
        ;

        RETURN COALESCE(flag, false);
    END IF;

    RETURN flag;
  END $$ LANGUAGE plpgsql STABLE;
--

--
CREATE OR REPLACE FUNCTION Tpersonal.bdep__individwt__get_date__by_bd_anf_minr(
      _bd_anf                         timestamp,
      _bd_minr                        integer,
      -- Datum der Gültigkeit (Setting s.u.) verwenden,
      -- sonst wird das ignoriert und Zeitpunkt des Schichtbeginns anhand des individ. Werktags ermittelt.
      _use_individwt_valid_from_date  boolean = true
  ) RETURNS timestamp AS $$
  DECLARE
      -- Datum (Setting), ab wann die Berechnung des individ. Werktags gültig ist, siehe #14870
      -- Fallback für Prüfung, wenn Setting null ist: nie gültig.
      _individwt_valid_from_date    CONSTANT date := coalesce( nullif( TSystem.Settings__Get('individwt_valid_from_date'), '' ), 'infinity');

      -- return value: Zeitpunkt des Schichtbeginns (enthält Datum des individ. Werktags)
      _individwt_shift_start_clock  timestamp;

  BEGIN
    -- Ermittlung des nächstgelegenen (<= gegebenen BD_ANF) Zeitpunkts des Schichtbeginns anhand bdep-Daten
    -- Gültigkeit anhand Start-Datum (Setting) prüfen, siehe #14870


    -- eigentliche Ermittlung des nächstgelegenen Zeitpunkts des Schichtbeginns (enthält Datum des individ. Werktag)
    _individwt_shift_start_clock :=
          bd_anf
        FROM bdep
        WHERE bd_minr = _bd_minr
          -- Performance: Einschränkung auf nur inhaltliche mögliche Daten (max. Tag davor)
          AND bd_anf::date BETWEEN _bd_anf::date - 1 AND _bd_anf::date
          -- Stempelungen danach irrelevant
          AND bd_anf <= _bd_anf
          -- Stempelung ist Schichtbeginn
          AND Tpersonal.bdep__individwt__isfirstofday( in_bdanf => bd_anf, in_bdminr => bd_minr )
        ORDER BY bd_anf DESC NULLS LAST
        LIMIT 1
    ;

    -- 1: Noch keine Stempelung vorhanden.
    -- 2: Aktuell angefragte Stempelung ist bereits Schichtbeginn, aber neuer Eintrag würde neuer Schichtbeginn werden (BEFORE INSERT).
    IF
            Tpersonal.bdep__individwt__isfirstofday( _bd_anf, _bd_minr )
        AND (
                _individwt_shift_start_clock IS NULL
            OR  _individwt_shift_start_clock < _bd_anf
        )

    THEN
        _individwt_shift_start_clock := _bd_anf;
    END IF;


    -- Gültigkeit anhand Start-Datum (Setting) prüfen, siehe #14870
    IF  -- Prüfung soll stattfinden (also nicht ignoriert werden). Default: ja.
            coalesce( _use_individwt_valid_from_date, true )

        -- Prüfung auf Ungültigkeit. Nur gültig, wenn individ. Werktag >= validem Datum ist.
        -- Wenn Setting nicht gesetzt ist, dann ist der individ. Werktag nie gültig.
        AND _individwt_shift_start_clock::date <  _individwt_valid_from_date

        -- Fallback für Ungültigkeit nur durchführen, wenn individ. Werktag vom Stempeldatum abweicht,
        -- sonst ist Zeitpunkt des Schichtbeginns heute korrekt.
        AND _individwt_shift_start_clock::date <> _bd_anf::date
    THEN
        -- Bei Ungültigkeit Fallback auf Stempelung
        _individwt_shift_start_clock := _bd_anf;

    END IF;


    RETURN coalesce( _individwt_shift_start_clock, _bd_anf );
  END $$ LANGUAGE plpgsql STABLE;
--

-- (Nach)Berechnen der individuellen Werktage aller noch nicht verbuchten Stempelungen.
CREATE OR REPLACE FUNCTION Tpersonal.bdep__individwt__recalc(
      in_newbdanf timestamp,
      in_oldbdanf timestamp,
      in_bdminr   integer
  )
  RETURNS         boolean
  AS $$
  DECLARE
      _rec_bdep    record;
      _result      boolean;
  BEGIN
    _result := false;

    FOR _rec_bdep IN
        SELECT bd_id,
               bd_anf,
               bd_individwt_mpl_date,
               Tpersonal.bdep__individwt__get_date__by_bd_anf_minr(
                   bd_anf,
                   bd_minr
               )::DATE AS individwt_neu
          FROM bdep
         WHERE bd_minr = in_bdminr
           AND NOT bd_buch
           AND bd_anf BETWEEN LEAST(in_newbdanf, in_oldbdanf) - INTERVAL '12 hours' AND GREATEST(in_newbdanf, in_oldbdanf) + INTERVAL '12 hours'
         ORDER BY bd_anf
    LOOP
        IF _rec_bdep.individwt_neu != _rec_bdep.bd_individwt_mpl_date THEN
           UPDATE bdep
              SET bd_individwt_mpl_date = _rec_bdep.individwt_neu
            WHERE bd_id = _rec_bdep.bd_id;

           -- wenn es Treffer gab, muss der Datensatz wegen einer evtl neuen Zuordnung auch neu berechnet werden. Siehe bdep__a_10_iud__individwt
           _result := true;
        END IF;
    END LOOP;

    RETURN _result;
  END $$ LANGUAGE plpgsql VOLATILE;
--

--Parameter:
--INOUT LookAhead - Anzahl der Tage die voraus geschaut werden soll
--OUT: Mitarbeiter-Nummer, Datum wann der Tag ist, Anzahl der Jahre die der Mitarbeiter da ist, Anzahl der Tagen bis Jubiläum
CREATE OR REPLACE FUNCTION tpersonal.personal__llv__GetStaffJubilee(INOUT LookAhead INTEGER = 7, OUT MiNr INTEGER, OUT Pers_dbrid VARCHAR, OUT bj_datum DATE, OUT bj_jahre INTEGER, OUT bj_tage INTEGER) RETURNS SETOF RECORD AS $$
  DECLARE rec RECORD;
  Daten VARCHAR;
 BEGIN
  Daten:=TRIM(TSystem.Settings__Get('MSG.Daten.Jubilee'));  --
  FOR rec IN SELECT
               (ll_einstd::timestamp + Date_Trunc('Year', AGE(ll_einstd::timestamp)) + '1 Year'::Interval)::Date - NOW()::Date AS days,
               ll_einstd,
               DATE_PART('year', AGE((ll_einstd - '1 Year'::Interval)::timestamp))::INTEGER AS years,
               personal.dbrid AS p_dbrid,
               ll_minr
             FROM personal
             LEFT OUTER JOIN llv ON ll_ad_krz=pers_krz
             WHERE
               DATE_PART('year', AGE((ll_einstd - '1 Year'::Interval)::timestamp))::INTEGER > 0 AND
               EXISTS(SELECT * FROM regexp_split_to_array(Daten, ';') WHERE regexp_split_to_array(Daten, ';') && ('{' || DATE_PART('year', AGE((ll_einstd - '1 Year'::Interval)::timestamp))::VARCHAR || '}')::TEXT[])
               --DATE_PART('year', AGE((ll_einstd - '1 Year'::Interval)::timestamp))::INTEGER % 5 = 0
               AND (ll_einstd::timestamp + Date_Trunc('Year', AGE(ll_einstd::timestamp)) + '1 Year'::Interval)::Date - NOW()::Date <= LookAhead
               AND COALESCE(ll_endd, current_date) >= current_date
             ORDER BY days LOOP
    MiNr       := rec.ll_minr;
    Pers_dbrid := rec.p_dbrid;
    bj_datum   := rec.ll_einstd;
    bj_jahre   := rec.years;
    bj_tage    := rec.days;
    RETURN NEXT;
  END LOOP;
  RETURN;
 END $$ LANGUAGE plpgsql;

--



CREATE OR REPLACE FUNCTION tpersonal.skillplan_create(datv DATE, datb DATE, alternative_for_id INTEGER) RETURNS VOID AS $$
    DECLARE _datid INTEGER;
    BEGIN
     INSERT INTO skillplan(dat_begin, dat_end) VALUES (datv, datb) RETURNING dat_id INTO _datid; --Grundsätzliche Anlage
     IF alternative_for_id IS NOT NULL THEN --die neue Schulung ist eine alternative aus, dann die Daten des Originals übernehmen
        UPDATE skillplan sa SET dat_altdat_for_dat_id=alternative_for_id, dat_subject=so.dat_subject, skp_due=so.skp_due, skp_ak_nr=so.skp_ak_nr, skp_lkn=so.skp_lkn, dat_location=so.dat_location
               FROM skillplan so WHERE so.dat_id=alternative_for_id AND sa.dat_id=_datid;
     END IF;
    END $$ LANGUAGE plpgsql;

-- Eintrag / Entfernen der Schulung aus dem Qualifikationsprofil der Mitarbeiter, F3
-- [offen] Fall, ungeklärt was mit Schulungen passiert, die bereits im Profil vorhanden sind
CREATE OR REPLACE FUNCTION tpersonal.skillplan_to_skillmitarbzu(mode INTEGER, skpid INTEGER, datv DATE DEFAULT NULL, datb DATE DEFAULT NULL, prufung BOOLEAN DEFAULT NULL, erfgrad NUMERIC DEFAULT NULL, theorie BOOLEAN DEFAULT NULL, praxis BOOLEAN DEFAULT NULL) RETURNS VOID AS $$
     DECLARE rec RECORD;
    BEGIN
    --Hole die Mitarbeiter zur geplanten Schulung
      FOR rec IN SELECT *, TRecnoParam.GetEnum('llv_members.skillplan.status', llv_members.dbrid) AS r_value FROM skillplan
        JOIN LATERAL (SELECT CASE WHEN dat_altdat_for_dat_id IS NULL THEN
                        skillplan.dbrid
                  ELSE
                     (SELECT so.dbrid FROM skillplan so WHERE so.dat_id=skillplan.dat_altdat_for_dat_id)
                  END AS skillplandbrid --wenn wir nur eine Alternative sind, dann ist die TeilnehmerListe immer unter dem Parent zu finden, Dokumente usw. ebenso)
                  ) AS skillplan_master ON true
            JOIN llv_members ON llm_parent_dbrid=skillplandbrid
            WHERE skp_id=skpid LOOP
      IF (mode=1) AND (rec.r_value='T') THEN --Insert erfolgt nur wenn Mitarbeiter auch teilgenommen hat
        INSERT INTO skillmitarbzu(smz_minr, smz_ak_nr, smz_lkn, smz_bdat, smz_edat, smz_prufung, smz_erfgrad, smz_theorie, smz_praxis, smz_txt, smz_descr)
          VALUES (rec.llm_minr, rec.skp_ak_nr, rec.skp_lkn, datv, datb, prufung, erfgrad, theorie, praxis, rec.dat_location, rec.dat_subject);
          --vorerst wird Ort in Zusatztext übergeben, da noch kein eigenes Feld;
          --Betreff dat_subject > Qualifikationsbeschreibung smz_descr Wunsch RSUAS
      END IF;
      IF (mode=-1) THEN
        DELETE FROM skillmitarbzu WHERE smz_ak_nr=rec.skp_ak_nr AND smz_minr=rec.llm_minr;
      END IF;
      END LOOP;
    END $$ LANGUAGE plpgsql;


-- #7502 Rechtsklickfunktion 'Zeiten im Auswahlbereich korrigieren' Modul Berabeiten : Präsenzzeit tabellarisch
CREATE OR REPLACE FUNCTION tpersonal.bdep__swaptime(minr INTEGER, bdanf TIMESTAMP, bdend TIMESTAMP) RETURNS VOID AS $$
    DECLARE r RECORD;
    BEGIN
      FOR r IN SELECT
        bd_id,
        (bd_anf::DATE||' '||bd_end::TIME)::TIMESTAMP AS new_bdanf, -- die Endezeit wird neue Anfangszeit ~ dies ist bereits der nächste Tag
        (bd_anf::DATE||' '||(lead(bd_anf) OVER(ORDER BY bd_anf, bd_id))::TIME)::TIMESTAMP AS new_bdend --die Anfangszeit des nächsten Satzes wird neue Endezeit des Vorgängers ~ lead holt den nächsten Datensatz auch hier muss 1 Tag abgezogen werden, da timestamp den nächsten Tag enthält
      FROM bdep
      WHERE bd_minr = minr AND bd_anf BETWEEN bdanf AND bdend ORDER BY bd_anf, bd_id LOOP
        UPDATE bdep SET bd_anf=r.new_bdanf, bd_end=r.new_bdend WHERE bd_id = r.bd_id AND r.new_bdend IS NOT NULL;
      END LOOP;
      --
      RETURN;
    END $$ LANGUAGE plpgsql;


/*Karenzzeit*/

CREATE OR REPLACE FUNCTION tpersonal.karenz_kr_newtime_from(TIMESTAMP, VARCHAR) RETURNS TIMESTAMP(0) WITHOUT TIME ZONE AS $$
   DECLARE newtime TIME(0) WITHOUT TIME ZONE;
    BEGIN
     newtime:=(SELECT kr_newtime FROM karenz WHERE kr_tpl_name=$2 AND CAST($1 AS TIME) BETWEEN kr_anf AND kr_end);
     IF newtime IS NOT NULL THEN
            RETURN CAST($1 AS DATE) || ' ' || newtime;
     ELSE
            RETURN $1;
     END IF;
    END$$LANGUAGE plpgsql STABLE;

 CREATE OR REPLACE FUNCTION Z_99_Deprecated.karenz_zeit(TIMESTAMP, VARCHAR) RETURNS TIMESTAMP(0) WITHOUT TIME ZONE AS $$
   SELECT tpersonal.karenz_kr_newtime_from($1, $2);
  $$ LANGUAGE sql;
--

-- Präsenzzeitstempelungen neu berechnen!
CREATE OR REPLACE FUNCTION tpersonal.mitpln__saldo__recalc__prsz_neu(
    in_anfdat DATE,
    in_enddat DATE,
    in_minr VARCHAR,
    in_saldonachrechnen BOOL DEFAULT false
    )
    RETURNS VOID
    AS $$
    DECLARE bdep_id INTEGER;
    BEGIN
      IF in_saldonachrechnen THEN -- Durch Setzen der Werte auf NULL, wird der Saldo neu berechnet, siehe mitpln__b_iu
          UPDATE mitpln SET
            mpl_min       = NULL,
            mpl_saldo     = NULL,
            mpl_absaldo   = NULL,
            mpl_tpl_name  = NULL,
            mpl_feiertag  = (SELECT ft_bez FROM feiertag WHERE ft_date = mpl_date AND NOT ft_urlaub)
          WHERE mpl_date BETWEEN in_anfdat AND in_enddat
            AND mpl_minr LIKE in_minr
            AND NOT mpl_buch;
      ELSE
          UPDATE mitpln SET
            mpl_min     = NULL,
            mpl_saldo   = NULL,
            mpl_absaldo = NULL
          WHERE mpl_date BETWEEN in_anfdat AND in_enddat
            AND mpl_minr LIKE in_minr
            AND NOT mpl_buch;
      END IF;

      -- Sollte das Folgende nicht vor der mitpln sein? DS - 2010-10-01

      -- Präsenzzeiten
      -- Update in korrekter Reihenfolge für Berücksichtigung ind. Werktag, vgl. #13917
      FOR bdep_id IN
          SELECT
            bd_id
          FROM bdep
          WHERE bd_individwt_mpl_date BETWEEN in_anfdat AND in_enddat
            AND bd_minr LIKE in_minr
            AND NOT bd_buch
          ORDER BY
            bd_minr, bd_anf, bd_id
      LOOP
          UPDATE bdep
             SET bd_gleitpause   = NULL,
                 bd_blockpause   = NULL,
                 bd_nachtstunden = NULL
           WHERE bd_id = bdep_id;
      END LOOP;

      -- Abwesenheiten Auslösen Trigger bdepab__b_iu erzwingen
      UPDATE bdepab
         SET bdab_id = bdab_id
       WHERE bdab_anf BETWEEN in_anfdat AND in_enddat
         AND bdab_minr = in_minr
        AND NOT bdab_buch;

      RETURN;
    END $$ LANGUAGE plpgsql VOLATILE;
--

--Für F2-Fenster Tagesplan auf Wochenbasis: BDE
CREATE OR REPLACE FUNCTION tpersonal.tplan_zeitraum_as_caption(IN tpName VARCHAR) RETURNS VARCHAR(50) AS $$
    DECLARE
      zbeg time;
      zend time;
    BEGIN
      IF tpName IS NOT NULL THEN
            SELECT tpl_rbegin, tpl_rend INTO zbeg, zend FROM tplan WHERE tpl_name=tpName;
            RETURN tpName||' ('||to_char(zbeg, 'HH24:MI')||'-'||to_char(zend, 'HH24:MI')||')';
      ELSE
            RETURN NULL;
      END IF;
    END $$ LANGUAGE plpgsql STABLE;

  CREATE OR REPLACE FUNCTION Z_99_Deprecated.getStandplanZeitraum(IN tpName VARCHAR) RETURNS VARCHAR(50) AS $$
    SELECT tpersonal.tplan_zeitraum_as_caption($1);
    $$ LANGUAGE sql;

-- Beginn Überstunden --

-- #8820 Ausgabe Überstunden-Salden eines Monats mit Ausweisung Normalüberstunden, Sonntag, Feiertag
-- Später evtl. Erweiterung notwendig für die Summierung der Sub-Daten über weitere Systemeinstellungen, zB 1 Summierung über alles, 2 nur Plus-Stunden
-- Einschränkung: nur Berücksichtigung der gestempelten Zeiten, da
-- + Berechnung unklar für Fall manuelle Änderung der Überstunden durch einen Personaler in der stuko_history
-- war tpersonal.uebersalden()
CREATE OR REPLACE FUNCTION tpersonal.llv__mitpln__uebersalden__get(
    IN  _minr  INTEGER,
    IN  _date  DATE,
    OUT gsaldo numeric,
    OUT ssaldo numeric,
    OUT fsaldo numeric,
    OUT nsaldo numeric
    )
    AS $$
    BEGIN

      -- ACHTUNG: bdep__saldo_vs_soll__by_monat__differenz__get macht fast das Gleiche. Müßte zusammengelegt werden.
      -- ACHTUNG: diese Funktion hier berücksichtigt ll_fixauszahlung nicht!!!! wäre zu prüfen und zu implementieren

        SELECT --das aktuelle Berechnungsschema geht davon aus, dass über alle Reihen summiert wird, negative Einträge den Saldo also reduzieren
               round(sum(saldo)   OVER (),2) AS saldosum, --Arbeitszeitplus/-minus
               round(sum(sosaldo) OVER (),2) AS sosaldosum, --Sonntagplus
               round(sum(fesaldo) OVER (),2) AS fesaldosum, --Feiertagplus

               --Normalüberstunden ~ enthalten keine Überstunden aus Sonntag, Feiertag
               CASE WHEN (sum(saldo) OVER ()) > (sum(sosaldo) OVER ()) + (sum(fesaldo) OVER ())
                    THEN round((sum(saldo) OVER ()) - (sum(sosaldo) OVER ()) - (sum(fesaldo) OVER ()), 2)
                    ELSE 0
               END AS normalsum
               --da Überstunden aus Nachtarbeit nicht erkannt werden können, werden diese hier nicht berücksichitgt #8143 ~ SUM(bd_nachtstunden) OVER () AS nachtsum,
          INTO gsaldo, ssaldo, fsaldo, nsaldo
          FROM
               (SELECT --Debug-Ansicht
                       -- CAST(mpl_date||'  '||dayNameOfWeek(mpl_date) AS CHAR(13)) AS day_date,
                       --Gesamtsaldo
                       mpl_saldo + mpl_absaldo - mpl_min AS saldo,
                       --Sonntag, kein Feiertag
                       CASE WHEN (extract('dow' FROM mpl_date) = 0) AND (mpl_feiertag IS null)
                            THEN mpl_saldo + mpl_absaldo - mpl_min
                            ELSE 0
                       END AS sosaldo,
                       --Feiertage
                       CASE WHEN (mpl_feiertag IS NOT null)
                            THEN mpl_saldo + mpl_absaldo - mpl_min
                            ELSE 0
                       END AS fesaldo
                 FROM mitpln
                      -- LEFT JOIN bdep ON mpl_minr=bd_minr AND mpl_date=timestamp_to_date(bd_anf) --benötigt für bd_nachtstunden
                WHERE mpl_minr = _minr
                  AND mpl_formonth = date_to_yearmonth_dec(_date)
                      -- Zuerst Freigabe prüfen, danach Verbuchung ~ nur verbuchte Zeiten sind feststehend und stimmig
                      -- heißt: Ich kann entweder verbuchte, oder freigegebene Zeiten zur Auszahlung beantragen
                  AND (    mpl_freigabe
                        OR mpl_buch
                           -- Überstundenauszahlung automatisch: immer alles auszahlen benötigt natürlich keine extra freigabe. Wird ja immer automatisch alles ausgezahlt
                        OR TSystem.Settings__GetBool('MonAbschlStukoN')
                       )
                ORDER BY mpl_date
               ) as innersel
         LIMIT 1; -- ?? Limit 1?! / fragt DS hä und wieso ??

        RETURN;
    END $$ LANGUAGE plpgsql STABLE;
--

-- #8722 Ausgabe Anzahl der Stunden, die der Mitarbeiter für Überstundenauszahlung beantragen darf
-- Implizit:  Wenn RETURN > 0 : Auszahlung möglich
-- IN firma zb Über TSystem.Settings__Get('KUNDE')
-- war tpersonal.stundauszahl_maxstunden()
CREATE OR REPLACE FUNCTION tpersonal.llv__stundauszahl__maxstunden__get(
      minr  integer,
      firma varchar = null
  ) RETURNS integer AS $$
  DECLARE
      _llstukobuch     numeric;  -- verbuchtes Stundenkonto
      _schwelle        integer;  -- Überstunden_schwelle, so viele Überstundnen muss der Mitarbeiter immer haben
      _rec_uebersalde  record;   -- Übersalden, siehe tpersonal.llv__mitpln__uebersalden__get
      _defdate         date;     -- Beantragungsdatum, siehe tpersonal.llv__stundauszahl___defDate__get
      _rpause          numeric;  -- Raucherpause summiert
      _maxstunden      integer;  -- Ergebnis

  BEGIN
      _llstukobuch := trunc( ll_stuko_buch ) FROM llv WHERE ll_minr = minr;  -- ? DG: trunc

      -- firmenspezifische Einstellungen
      IF firma NOT IN ( 'LOLL', 'LOLL-TEST', 'DEMO' ) THEN
          RETURN _llstukobuch;
      END IF;

      _schwelle        := TSystem.Settings__GetInteger( 'BDE_STUNDAUSZAHL_SCHWELLE' );
      _rec_uebersalde  := tpersonal.llv__mitpln__uebersalden__get( minr, current_date );
      _defdate         := ( tpersonal.llv__stundauszahl__defDate__get( minr ) ).defDate;

      -- #15155: nur für den Fall, dass ein tatsächliches Datum ermittelt werden konnte
      -- Raucherpausen abfragen. Raucherpausenabfrage ohne Datumsfelder führt zu Exception
      IF _defdate IS NOT NULL THEN

          _rpause :=
              tpersonal.llv__bdpab__rpause__round(
                minr,
                date__first_day_of_month__get( _defdate ),
                date__last_day_of_month__get( _defdate )
              )
          ;
      END IF;

      -- Auszahlungssfähig sind:
      _maxstunden :=
          -- TRUNC : Festsetzung bzw Vorgabe weil sichheitshalber der volle, abgerundete auszahlbare Stundenbetrag gewertet werden soll
          trunc(
              _llstukobuch                               --   verbuchtes Stundenkonto
            + coalesce( _rec_uebersalde.gsaldo, 0 )      -- + freigegebene Überstunden
            - coalesce( _schwelle, 0 )                   -- - Überstundenschwelle
            - coalesce( _rpause, 0 )                     -- - Raucherpause
            - (
                  coalesce( _rec_uebersalde.ssaldo, 0 )  -- - Sonn- und Feiertagsstunden (diese werden zwangsweise ausgezahlt)
                + coalesce( _rec_uebersalde.fsaldo, 0 )
              )
          )
        ;

      -- nie unter 0
      RETURN greatest( _maxstunden, 0 );

  END $$ LANGUAGE plpgsql STABLE;
--

 -- #9566 Monatsabschluss: automtatische Überstundenauszahlung erstellen
 -- war tpersonal.stundauszahl_upsert
CREATE OR REPLACE FUNCTION tpersonal.llv__stundauszahl__beantragung__create(
    IN _minr integer,
    IN _date date
    )
    RETURNS void
    AS $$
    DECLARE _stundauszahl_recordcount integer;
            _saldo_sunday numeric;
            _saldo_holiday numeric;
            _saldo_norm numeric;
    BEGIN
        -- initiale Prüfungen: Anzahl Datensätze korrekt usw
          _stundauszahl_recordcount := (SELECT count(1)
                                          FROM stundauszahl
                                         WHERE sa_minr = _minr
                                           AND date_to_yearmonth(sa_date) = date_to_yearmonth(_date)
                                           -- ausgenommen Urlaubsfall ~ gleiche Tabelle (Urlaubsauszahlung)
                                           AND sa_type = 'hours'
                                        );

          IF _stundauszahl_recordcount > 1 THEN
             -- automatische Überstunden geht nur von einem automatischen Eintrag aus. Wenn hier 2 Datensätze sind, muss geklärt werden wieso und
              --dies evtl im Algorithmus berücksichtig werden. siehe unten: das update geht pauschal auf alles mit diesem monat
             RAISE EXCEPTION 'invalid data: more than one record foun in table stundauszahl. expect 0 or 1: %', _stundauszahl_recordcount;
          END IF;
        --

        -- Ermitteln der Überstunden
        SELECT coalesce(ssaldo, 0),
               coalesce(fsaldo, 0),
               coalesce(nsaldo, 0)
          INTO _saldo_sunday,
               _saldo_holiday,
               _saldo_norm
          FROM tpersonal.llv__mitpln__uebersalden__get(_minr, _date)
        ;


        -- wir haben keine (automatisch auszuzahlenden) überstunden. Dieser Datensatz würde in stundauszahl__a_iu sofort wieder gelöscht werden
        IF     _saldo_sunday = 0
           AND _saldo_holiday = 0
        THEN
           RETURN;
        END IF;

        -- Insert-Fall
         -- Automatismus, wenn Mitarbeiter nichts zusätzlich beantragt
         -- Nur Einfügen Wenn Sonntagstunden + Feiertagsstunden vorhanden sind ODER generelle Auszahlung
         -- Nur Sonn- und Feiertagsstunden müssen zwangsweise ausbezahlt werden
        IF _stundauszahl_recordcount = 0 THEN
           INSERT INTO stundauszahl
                       (sa_minr,
                        sa_date,
                        sa_sonntag,
                        sa_feiertag,
                        sa_normal,
                        sa_type
                       )
                VALUES (_minr,
                        _date,
                        _saldo_sunday,
                        _saldo_holiday,
                        _saldo_norm,
                        'hours'
                       )
           ;
           --WHERE nsaldo > 0;  -- Wenn Normalstunden > 0 Dann sind tatsächlich Überstunden vorhanden
   ELSE
           --  Bei 2 Datensätzen den richtigen treffen : stundauszahl__b_i
           UPDATE stundauszahl
              SET sa_sonntag  = _saldo_sunday,
                  sa_feiertag = _saldo_holiday,
                  sa_normal   = _saldo_norm
            WHERE sa_minr = _minr
              AND date_to_yearmonth(sa_date) = date_to_yearmonth(_date)
              AND NOT sa_buch
                  -- Urlaub ist ein eigener Datensatz
              AND sa_type = 'hours'
           ;

           IF NOT FOUND THEN
              RAISE EXCEPTION 'internal error: stundauszahl record not found. sa_buch is true?! and should not be true!';
           END IF;
        END IF;

        -- ungeklärt: Wenn weder Ins noch Upd Dann Meldung an Nutzer?
        RETURN;
        -- Update sa_stunden via stundauszahl__a_iu
    END $$ LANGUAGE plpgsql;


-- #8143 Prüfung Beantragung ist möglich; Wenn Ja, Ausgabe des Beantragungs-Datum / Wenn Nein, Ausgabe Text
-- Firmenspezifische Steuerung ist später möglich : Tagesschwelle aktuell fest 3
-- Prüfung Jahreswechsel funktioniert nicht, wenn mehr als 1 Jahr kein Monatsabschluss gemacht wurde: das nehmen wir aber als gesetzt an
-- war tpersonal.stundauszahl_defDate
CREATE OR REPLACE FUNCTION tpersonal.llv__stundauszahl__defDate__get(minr INTEGER, firma VARCHAR DEFAULT NULL, OUT defDate DATE, OUT usrinfo Text) RETURNS RECORD AS $$
    DECLARE llshmonthyear VARCHAR;
            novprevyear VARCHAR; --November letztes Jahr
            decprevyear VARCHAR; --Dezember letztes Jahr
            monthdiff INTEGER;
    BEGIN
     -- Mitarbeiter darf beantragen?
     IF (SELECT ll_sa_request FROM llv WHERE ll_minr=minr) THEN
       llshmonthyear:=(SELECT llsh_monthyear FROM llv_stuko_history WHERE llsh_ll_minr=minr ORDER BY llsh_monthyear DESC LIMIT 1);
       novprevyear:=((date_part('year', current_date)-1)::VARCHAR||'10'); --201810 ab 01.01.2019
       decprevyear:=((date_part('year', current_date)-1)::VARCHAR||'11'); --201811 ab 01.01.2019
       monthdiff:=(date_part('month', current_date)::INT) - (substring(llshmonthyear from 5 for 6)::INT+1);

       -- Beantragung ist im Anschluss an den letzten verbuchten Monat
       IF (monthdiff = 1) --Normalfall: Differenz aktueller Monat (dieses Jahres) - Monat letzer Monatsabschluss (diesen Jahres) = 1
           OR ( (llshmonthyear=decprevyear) AND (monthdiff = -11) ) THEN --Jahreswechsel: zb Januar-Dezember : 1-12
         defDate:=last_day(current_date); --wegen Monatsabschluss-Automatismus für letzten Tag des Monats #11976
         usrinfo:=NULL; --kein xtt, da Beantragung möglich
         RETURN;
       END IF;
       -- Beantragung ist bereits im nächsten Monat, Vormonat steht kurz vor der Verbuchung
       -- Vorsichtshalber bei Abstand 2 auch die Vorjahresverbuchung von November prüfen
       -- 3.Tag des Monats ist Regel von LOLL
       IF ( (monthdiff = 2) --Normalfall: (Differenz aktueller Monat (dieses Jahres) - Monat letzer Monatsabschluss (diesen Jahres) = 2)
             OR ((llshmonthyear=novprevyear) AND (monthdiff = -10))  --Jahreswechsel: zb Januar-November : 1-11
             OR ((llshmonthyear=decprevyear) AND (monthdiff = -10))) --Jahreswechsel: zb Februar-Dezember : 2-12
           AND ( (date_part('day', current_date)::INT) <= COALESCE(TSystem.Settings__GetInteger('BDE_STUNDAUSZAHL_KARENZTAGE'),0) ) THEN   --Bis zum 3. Tag des nächsten Monats
         defDate:= last_day( (current_date - '1 Month'::Interval)::Date ); -- Es soll für den Vormonat beantragt werden; --wegen Monatsabschluss-Automatismus für letzten Tag des Monats #11976
         usrinfo:=NULL; --kein xtt, da Beantragung möglich
       ELSE -- Keine Beantragung möglich
         defDate:=NULL;
         -- Beantragung ist bereits im nächsten Monat, Vormonat steht kurz vor der Verbuchung
         IF (monthdiff = 2) --Normalfall: (Differenz aktueller Monat (dieses Jahres) - Monat letzer Monatsabschluss (diesen Jahres) = 2)
             OR ((llshmonthyear=novprevyear) AND (monthdiff = -10))
             OR ((llshmonthyear=decprevyear) AND (monthdiff = -10)) THEN  --Jahreswechsel: zb Februar-Dezember : 2-12
           usrinfo:=lang_text(25421); -- 'Keine Bearbeitung möglich: Bearbeitungsfrist ist überschritten';
         END IF;
         -- Vormonat ist nicht verbucht
         IF (monthdiff > 2) --Normalfall
             OR ((llshmonthyear=decprevyear) AND (monthdiff > -10)) THEN  --Jahreswechsel
           usrinfo:=lang_text(25422); -- 'Keine Bearbeitung möglich: Vormonat ist nicht verbucht';
         END IF;
       END IF;
     ELSE
       defDate:=NULL;
       usrinfo:=lang_text(25423); -- 'Keine Bearbeitung möglich: Der Mitarbeiter verfügt nicht über die erforderlichen Rechte';
     END IF;
     RETURN;
    END $$ LANGUAGE plpgsql STABLE;



-- Ende Überstunden --

-- Begin Monatsabschluss --

-- Gibt True, wenn siehe prevYear-Kommentar (zuerst für #5106 benötigt)
CREATE OR REPLACE FUNCTION tpersonal.llv_stuko_history__checkYearDone(minr INTEGER, prevYear BOOLEAN DEFAULT TRUE, date DATE DEFAULT current_date) RETURNS BOOLEAN AS $$
    BEGIN
     -- Gibt True, wenn der Monatsabschluss des Vorjahres erledigt ist
     IF prevYear THEN
       RETURN COALESCE( (SELECT true FROM llv_stuko_history
         WHERE llsh_ll_minr=minr AND llsh_monthyear=((date_part('year', date)-1)::VARCHAR||'11'))
         , false);
     END IF;
     --Gibt true, das EingabeDatum im Dezember ist UND Dezember noch nicht verbucht ist
     IF NOT prevYear THEN -- wir benötigen das Jahr des Monatsabschluss
       RETURN COALESCE( (SELECT true FROM llv_stuko_history
         WHERE llsh_ll_minr=minr
         AND extract(month FROM date) =12 -- Monatsabschluss im Dezember // v2 prüfung auf Nov llsh_monthyear=((date_part('year', date))::VARCHAR||'10')
         AND NOT EXISTS ( SELECT true FROM llv_stuko_history WHERE llsh_ll_minr=minr AND llsh_monthyear=((date_part('year', date))::VARCHAR||'11') )
         LIMIT 1)
         , false);
     END IF;
    END $$ LANGUAGE plpgsql;

-- End Monatsabschluss --

-- Beginn Urlaubsberechnungen --

-- Restanspruch Vorjahr: Wähle den letzten Urlaubseintrag des letzten Jahres im Monatsabschluss #6883, #8944, #12682
-- Parameter EndeDatum aus get_urlaubstage fehlt noch
CREATE OR REPLACE FUNCTION tpersonal.llv__urlaub__rest_vorjahr(
    IN _minr integer,
    IN _NurWochentage boolean DEFAULT False,
    IN _date date DEFAULT current_date
    )
    RETURNS numeric
    AS $$
    DECLARE _llshurlaub numeric;
             _curryear boolean;
             _llvstukoexists boolean;
    BEGIN
        -- Urlaubskonto Vorjahr Monat Dezember (Index 11) ermitteln
        _llshurlaub := (SELECT coalesce(llsh_urlaub, 0)
                          FROM llv_stuko_history
                         WHERE llsh_monthyear = ( date_part('year', _date) - 1 )::varchar || 11 -- Vorjahr mit Monat 11
                           AND llsh_ll_minr = _minr
                        );

        -- Festestellen, ob ein Monatsabschluss in diesem Jahr vorliegt.
        _curryear := EXISTS (SELECT true FROM llv_stuko_history
                              WHERE -- Jahr aus YYYYMM extrahieren und vergleichen mit aktuellem Jahr
                                    ( substring(llsh_monthyear from 1 for 4) )::varchar = (date_part('year', _date))::varchar
                                AND llsh_ll_minr = _minr
                      );

        -- Feststellen, ob es überhaupt bereits eine Monatsverbuchung gibt
        _llvstukoexists := (SELECT EXISTS(SELECT true FROM llv_stuko_history WHERE llsh_ll_minr = _minr LIMIT 1));

        --
        IF (_llshurlaub IS NOT NULL) THEN
           -- Vorjahr ist vollständig abgeschlossen, dann geben wir den Stand aus Dezember Vorjahr zurück
           RETURN _llshurlaub;
        ELSEIF (_curryear OR NOT _llvstukoexists) THEN
           -- Der erste Monatsabschluss ist im aktuellen Jahr : Neueinführung der Präsenzzeit -> es gibt keinen Resturlaub aus Vorjahr
              -- Fall: im Vorjahr sind Testdaten angelegt. Diese waren nicht verbucht. Somit gibt es keinen Vorjahresabschluss.
              --       damit diese im nächsten Else nicht doch gezogen werden, prüfen wir: kein Vorjahresabschluss, aber ein Dies-Jahresabschluss, somit auch kein Urlaub Vorjahr, was llv__urlaub__rest__by__futuredate__get sonst errechnen würde
           -- Oder es gibt noch keinen Eintrag in den Monatsabschlüssen -> Mitarbeiter neu angelegt
           RETURN 0;
        ELSE
           -- Sonst den letzten verbuchten Urlaub abzüglich nicht verbuchter Urlaube bis zum Vor-Jahresende >> 1.1.aktuelles Jahr
           RETURN tpersonal.llv__urlaub__rest__by__futuredate__get(
                        _minr,
                        _NurWochentage,
                        -- gibt 01.01. aktuelles Jahr zurück
                        date_trunc('year', _date)::date
                  );
        END IF;
    END $$ LANGUAGE plpgsql STABLE;


-- z_99_deprecated.set_llv_urlaubges
CREATE OR REPLACE FUNCTION tpersonal.llv__urlaub_ges__from__anspruch__set(yyyy integer DEFAULT NULL, minr integer DEFAULT NULL) RETURNS VOID AS $$
 DECLARE curryear integer;
 BEGIN
     curryear := date_part('year', current_date);
     -- Jahresurlaub MitVz setzen
     UPDATE llv
        SET ll_urlaub_ges = uan_urlaub_ges
       FROM llv_url_anspr
      WHERE uan_ll_minr = ll_minr
        AND NOT uan_done
        AND ll_minr  = coalesce(minr,ll_minr) -- nur dieser Mitarbeiter für Monatsabschluss
        AND uan_year = coalesce(yyyy, curryear) AND uan_year >= curryear; --Zusatzsicherungen
     -- Erledigt-Status Staffel setzen (wird nach Ablaufänderung #5106 eigentlich nicht mehr benötigt)
     UPDATE llv_url_anspr
        SET uan_done = True
      WHERE uan_ll_minr = coalesce(minr,uan_ll_minr) -- nur dieser Mitarbeiter für Monatsabschluss
        AND uan_year    = coalesce(yyyy, curryear)
        AND uan_year >= curryear; --Zusatzsicherungen
     --
     RETURN;
 END $$ LANGUAGE plpgsql;



--Anspruch gesamt (aktuelles Jahr) #8944
--Parameter EndeDatum aus get_urlaubstage fehlt noch
CREATE OR REPLACE FUNCTION tpersonal.llv__urlaub__anspruch(
    IN _minr integer,
    IN _NurWochentage boolean DEFAULT False
    )
    RETURNS numeric
    AS $$
        -- Resturlaub (durch Monatsabschluss des Vorjahres) ODER letzten verbuchten Urlaub abzüglich nicht verbuchter Urlaube bis zum Vor-Jahresende + Jahresurlaub
        SELECT    coalesce(tpersonal.llv__urlaub__rest_vorjahr(_minr, _NurWochentage), 0)
               +  (SELECT ll_urlaub_ges FROM llv WHERE ll_minr = _minr);
    $$ LANGUAGE sql STABLE;


-- Aktueller Urlaubsanspruch, Berücksichtigung Fall auch ohne Vor-Jahresabschluss #8944, #12682
-- date wegen Testfähigkeit hinzugefügt #20357
-- Erweiterung #22296 für 2 zurückliegende Jahre, wenn kein Monatsabschluss, bspw Sonderfall Jahreswechsel
CREATE OR REPLACE FUNCTION tpersonal.llv__urlaub__act(
    IN _minr integer,
    IN _NurWochentage boolean DEFAULT False,
    IN _date date DEFAULT current_date
    )
    RETURNS numeric
    AS $$
    DECLARE _llurlaubges numeric;
            _llurlaub NUMERIC;
            _llendd DATE;

            _llvstukoexists BOOLEAN;
            _stukohistory_of_date_exists BOOLEAN;
            _stukohistory_of_date_prev_year_exists BOOLEAN;
            _stukohistory_of_date_prev_year_2_exists BOOLEAN;
            _stukohistory_after_date_exists BOOLEAN;
            _latest_monthyear VARCHAR;

            _last_recorded_urlaub NUMERIC DEFAULT 0;
            _last_month_date DATE;
    BEGIN
      -- Benötigte Status in einem Aufruf
      SELECT
        true,
        (substring(llsh_monthyear FROM 1 FOR 4) = date_part('year', _date)::VARCHAR), -- Monatsabschluss des Eingabedatum-Jahres vorhanden
        (llsh_monthyear   =  (date_part('year', _date) - 1)::varchar ||  11),         -- Monatsabschluss Dezember Vorjahr -> Jahresurlaub für Eingabedatum-Jahr wurde eingetragen
        (llsh_monthyear   =  (date_part('year', _date) - 2)::varchar ||  11),         -- Monatsabschluss Dezember VorVorjahr -> Jahresurlaub wurde eingetragen
        (yearmonth_dec_to_date(llsh_monthyear::INTEGER, true) > _date),               -- max. Monatsabschluss > Eingabedatum
        llsh_monthyear
      INTO
        _llvstukoexists,
        _stukohistory_of_date_exists,
        _stukohistory_of_date_prev_year_exists,
        _stukohistory_of_date_prev_year_2_exists,
        _stukohistory_after_date_exists,
        _latest_monthyear
      FROM llv_stuko_history
      WHERE llsh_ll_minr = _minr
      ORDER BY llsh_monthyear DESC
      LIMIT 1;

      IF NOT FOUND THEN
        _llvstukoexists := false;
        _stukohistory_of_date_exists := false;
        _stukohistory_of_date_prev_year_exists := false;
        _stukohistory_of_date_prev_year_2_exists := false;
        _stukohistory_after_date_exists := false;
        _latest_monthyear := NULL;
      END IF;

      SELECT ll_urlaub_ges, ll_urlaub, ll_endd FROM llv WHERE ll_minr=_minr
        INTO _llurlaubges, _llurlaub, _llendd;

      IF (coalesce(_llendd, _date) < _date) THEN
        RETURN _llurlaub;
      ELSE
        IF (    _stukohistory_of_date_exists
             OR _stukohistory_of_date_prev_year_exists
             OR NOT _llvstukoexists -- Oder es gibt noch keinen Eintrag in den Monatsabschlüssen -> Mitarbeiter neu angelegt
            )
        THEN
          -- 'verbuchter Resturlaub' abzüglich 'Urlaub genommen'
          RETURN tpersonal.llv__urlaub__rest__by__futuredate__get(_minr, _NurWochentage, _date);
        ELSE
          -- futuredate-Fall : das Eingabedatum liegt nach dem letzten Monatsabschluss
          IF ( NOT _stukohistory_after_date_exists ) THEN
            -- Es wird der (noch nicht eingetragene) Jahresanspruch addiert
              IF ( substring(_latest_monthyear FROM 1 FOR 4) = date_part('year', (_date - interval '1 year') )::VARCHAR ) -- Abstand 1 Jahr
                  OR _stukohistory_of_date_prev_year_2_exists THEN
                RETURN (tpersonal.llv__urlaub__rest__by__futuredate__get(_minr, _NurWochentage, _date)) + _llurlaubges;
              ELSE --Abstand zum Jahreswechsel mehr als 1 Jahr und Vorjahr nicht verbucht, weitere Fälle voerst nicht betrachten
                RETURN (tpersonal.llv__urlaub__rest__by__futuredate__get(_minr, _NurWochentage, _date)) + _llurlaubges + _llurlaubges ;
              END IF;
          ELSE
            -- Es gibt ältere Verbuchungen und man kann nicht über das darüberstehende SELECT auf das letzte stukohistory-element gehen
            -- Der zum Eingabedatum gültige Urlaubsanspruch berechnet sich hier über den verbuchten Wert im letzten Monat vor dem Eingabedatum abzüglich der Urlaubstage vom 01. des Folgemonats bis zum Eingabedatum
            RETURN tpersonal.llv__urlaub__rest__by__pastdate__get(_minr, _date, _NurWochentage);
          END IF;
        END IF;
      END IF;
    END $$ LANGUAGE plpgsql STABLE;


-- Verplante Abwesenheiten bis Jahresende: Alle genehmigten Urlaube des laufenden Jahres ab heute bis Jahresende #6883.
-- Hinweis: Urlaub 1/2 ist Urlaub und muss über id 1 abgefragt werden
CREATE OR REPLACE FUNCTION tpersonal.llv__abw__geplant__by__currentyear__get(
    IN _minr integer,
    IN _beantragung_inkl boolean DEFAULT false,
    IN _beantragung_nur boolean DEFAULT false,
    IN _date date = current_date,
    IN _abw_id integer DEFAULT 1 -- Default Abwesenheit Urlaub
    )
    RETURNS numeric(12,2)
    AS $$
    BEGIN
      --Definition für den Zeitraum 'geplanter Abwesenheiten' : das In-Date ist synonym mit heute als abgeschlossen zu betrachten, berücksichtigt wird ab dem nächsten Tag
      RETURN
        COALESCE( ifThen( _beantragung_nur, 0, null), -- wenn _beantragung_nur, die genehmigten Urlaubstage ignorieren
        (
          -- Urlaub ganze Tage
          SELECT SUM (-- ohne Feiertage und Wochenenden als Zeitspanne
                        case
                          when bdab_anf <= _date and bdab_end > _date then
                            timediff( _date + 1, bdab_end, true, true, true ) -- Das In-Date berührt das Anfangsdatum der Abwesenheit, berücksichtigt wird ab dem nächsten Tag
                          when bdab_end = _date then
                            0 --das Ende-Datum der Abwesenheit ist am In-Date und somit bereits abgeschlossen, keine Berücksichtigung der kompletten Abwesenheit
                          else
                            timediff( bdab_anf, bdab_end, true, true, true )
                        end
                     )
          FROM bdepab
          WHERE
                  bdab_minr = _minr
              AND bdab_aus_id = _abw_id -- id der Abwesenheit
                  -- Abwesenheiten, die in der Zukunft liegen oder die unseren Zeitraum noch berühren
              AND bdab_end::date >= _date
              AND tpersonal.vacation__in__rest__of__employees__year__is( _minr, bdab_anf, bdab_end, _date )
        )
        , 0)
        +
        COALESCE(ifThen( _beantragung_nur, 0, null),
        (
          -- Urlaub 1/2 Tage
          SELECT SUM (-- ohne Feiertage und Wochenenden als Zeitspanne
                        case
                          when bdab_anf <= _date and bdab_end > _date then
                            timediff( _date + 1, bdab_end, true, true, true ) -- Das In-Date berührt das Anfangsdatum der Abwesenheit, berücksichtigt wird ab dem nächsten Tag
                          when bdab_end = _date then
                            0 --das Ende-Datum der Abwesenheit ist am In-Date und somit bereits abgeschlossen, keine Berücksichtigung der kompletten Abwesenheit
                          else
                            timediff( bdab_anf, bdab_end, true, true, true )
                        end
                     ) * 0.5
          FROM bdepab
          WHERE
                  bdab_minr = _minr
              AND bdab_aus_id = 100 -- Urlaub 1/2
              AND _abw_id = 1 --nur Abwesenheit Urlaub geprüft wird
              AND bdab_end::date >= _date
              AND tpersonal.vacation__in__rest__of__employees__year__is( _minr, bdab_anf, bdab_end, _date )
        )
        , 0)
        +
        COALESCE(ifThen( _beantragung_nur, 0, null ),
        (
          -- Betriebsurlaub (wird bei Option _beantragung_nur nicht eingerechnet, sonst ja)
          SELECT count(ft_date)
          FROM feiertag, llv
          WHERE
                  ft_urlaub -- ist Betriebsurlaub
              AND (ft_date BETWEEN _date + 1 AND tpersonal.end__of__employees__year__get( _minr, _date )) -- Berücksichtigt werden alle BU ab dem nächsten Tag (Festlegung) bis 31.12.
              AND EXTRACT(dow FROM ft_date) NOT IN (0, 6) -- Absicherung, dass Betriebsurlaub nicht auf Wochende liegt
              AND NOT EXISTS( SELECT true FROM bdepab WHERE bdab_minr = _minr
                                                        AND bdab_aus_id IN (1, 100)
                                                        AND ft_date between bdab_anf AND bdab_end) -- Wenn eingetragener Urlaub an dem Tag, dann diesen zählen und BU ignorieren (nicht doppelt).
              AND ll_minr = _minr
              AND ll_urlaub_betriebsurlaub_calc
        )
        , 0)
        +
        COALESCE(
        (
          -- Beantragungen Urlaub ganze Tage
          SELECT SUM (-- ohne Feiertage und Wochenenden als Zeitspanne
                        case
                          when bdabb_anf <= _date and bdabb_end > _date then
                            timediff( _date + 1, bdabb_end, true, true, true ) -- Das In-Date berührt das Anfangsdatum der Abwesenheit, berücksichtigt wird ab dem nächsten Tag
                          when bdabb_end = _date then
                            0 --das Ende-Datum der Abwesenheit ist am In-Date und somit bereits abgeschlossen, keine Berücksichtigung der kompletten Abwesenheit
                          else
                            timediff( bdabb_anf, bdabb_end, true, true, true )
                        end
                     )
          FROM bdepabbe
          WHERE
                  bdabb_minr = _minr
              AND bdabb_aus_id = _abw_id -- id der Abwesenheit
              AND NOT bdabb_bewill
              AND NOT bdabb_ablehn -- nicht bewillgt, aber auch nicht abgelehnt
              AND bdabb_end::date >= _date
              AND tpersonal.vacation__in__rest__of__employees__year__is( _minr, bdabb_anf, bdabb_end, _date )
              AND ( _beantragung_inkl OR _beantragung_nur )
        )
        , 0)
        +
        COALESCE(
        (
          --Beantragungen Urlaub 1/2 Tage
          SELECT SUM (-- ohne Feiertage und Wochenenden als Zeitspanne
                        case
                          when bdabb_anf<= _date and bdabb_end> _date then
                            timediff( _date + 1, bdabb_end, true, true, true ) -- Das In-Date berührt das Anfangsdatum der Abwesenheit, berücksichtigt wird ab dem nächsten Tag
                          when bdabb_end = _date then
                            0 --das Ende-Datum der Abwesenheit ist am In-Date und somit bereits abgeschlossen, keine Berücksichtigung der kompletten Abwesenheit
                          else
                            timediff( bdabb_anf, bdabb_end, true, true, true )
                        end
                     ) * 0.5
          FROM bdepabbe
          WHERE
               bdabb_minr = _minr
              AND bdabb_aus_id = 100 -- Urlaub 1/2
              AND _abw_id = 1 --nur Abwesenheit Urlaub geprüft wird
              AND NOT bdabb_bewill
              AND NOT bdabb_ablehn -- nicht bewillgt, aber auch nicht abgelehnt
              AND bdabb_end::date >= _date
              AND tpersonal.vacation__in__rest__of__employees__year__is( _minr, bdabb_anf, bdabb_end, _date )
              AND ( _beantragung_inkl OR _beantragung_nur )
        )
        , 0);
    END $$ LANGUAGE plpgsql STABLE;
--

-- Verplanter Urlaub bis Jahresende: Alle genehmigten Urlaube des laufenden Jahres ab heute bis Jahresende ursprünglich #6883.
CREATE OR REPLACE FUNCTION tpersonal.llv__urlaub__geplant__by__currentyear__get(
    IN _minr integer,
    IN _beantragung_inkl boolean DEFAULT false,
    IN _beantragung_nur boolean DEFAULT false,
    IN _date date = current_date
    )
    RETURNS numeric(12,2)
    AS $$
        SELECT tpersonal.llv__abw__geplant__by__currentyear__get(_minr => _minr,
                                                                 _beantragung_inkl => _beantragung_inkl,
                                                                 _beantragung_nur => _beantragung_nur,
                                                                 _date => _date,
                                                                 _abw_id => 1 --fix Urlaub
                                                                );
    $$ LANGUAGE sql STABLE;


CREATE OR REPLACE FUNCTION tpersonal.llv__urlaub__geplant__beantragt__by__currentyear__get(
    IN _minr integer,
    IN _date date DEFAULT current_date
    )
    RETURNS numeric(12,2)
    AS $$
        SELECT tpersonal.llv__urlaub__geplant__by__currentyear__get(_minr => _minr, _beantragung_inkl => false, _beantragung_nur => true, _date => _date)
    $$ LANGUAGE sql STABLE;


CREATE OR REPLACE FUNCTION tpersonal.llv__urlaub__planbar__by__currentyear__get(
    IN _minr integer,
    IN _date date DEFAULT current_date
    )
    RETURNS numeric(12,2)
    AS $$
        -- aktueller Urlaub minus geplanter UND beantragter Urlaub
        SELECT   tpersonal.llv__urlaub__act(_minr, true, _date) -- nur Wochentage true
               - tpersonal.llv__urlaub__geplant__by__currentyear__get(_minr, _beantragung_inkl => True, _date => _date) -- um date erweitert 19405 wegen testfähigkeit
    $$ LANGUAGE sql;

--Wichtigste Kennzahlen Stundenkonto der letzten Verbuchung
CREATE OR REPLACE FUNCTION tpersonal.llv_stuko_history_last(IN minr INTEGER, OUT monthyear VARCHAR, OUT llsh_urlaub NUMERIC, OUT llsh_stuko NUMERIC, OUT llsh_id INTEGER) RETURNS RECORD AS $$
DECLARE r RECORD;
BEGIN
--
  SELECT to_char( ('01.'||((substring(llsh_monthyear from 5 for 6)::INT)+1)::VARCHAR||'.'||(substring(llsh_monthyear from 1 for 4)))::DATE, 'TMMonth YYYY' ) AS monthyear,
         sh.llsh_urlaub, sh.llsh_stuko,
         sh.llsh_id
    INTO monthyear, llsh_urlaub, llsh_stuko, llsh_id
    FROM llv_stuko_history sh
   WHERE llsh_ll_minr = minr
   ORDER BY llsh_monthyear DESC LIMIT 1;
  RETURN;

END $$ LANGUAGE plpgsql STABLE;

-- Ende Urlaubsberechnungen --

--- #9579 MiNr change. Variante 'MiNrZusammenFuehren' braucht man zu klären
CREATE OR REPLACE FUNCTION tpersonal.llv__ll_minr_update__cascade(IN minr_old INTEGER, IN minr_new INTEGER, IN MiNrZusammenFuehren BOOLEAN DEFAULT false) RETURNS VOID AS $$
  DECLARE ll_adkrz_old VARCHAR;
  db_usename_old VARCHAR;
  db_usename_new VARCHAR;
  BEGIN
    IF NOT MiNrZusammenFuehren THEN
      --Wenn Metarbeiter-Nr existiert => Fehler.
      IF EXISTS(SELECT true FROM public.llv WHERE ll_minr = minr_new) THEN
            RAISE EXCEPTION '%', lang_text(29283);  -- Mitarbeiter-Nr existiert bereits
        END IF;
    ELSE
        SELECT ll_ad_krz INTO ll_adkrz_old, db_usename_old FROM llv WHERE ll_minr = minr_old;
        SELECT ll_ad_krz INTO db_usename_old FROM llv WHERE ll_minr = minr_new;
    END IF;

    -- Audit-Log abschalten: so tun, als wären wir 'syncro' (u.a. wg. modified-Triggern)
    SET LOCAL SESSION AUTHORIZATION 'syncro';

    PERFORM TSystem.trigger__disable( 'maschausf' );
    PERFORM TSystem.trigger__disable( 'stundauszahl' );
    PERFORM TSystem.trigger__disable( 'ExportLohn' );
    PERFORM TSystem.trigger__disable( 'bdep_log' );
    PERFORM TSystem.trigger__disable( 'bdepftbe' );
    PERFORM TSystem.trigger__disable( 'personal' );
    PERFORM TSystem.trigger__disable( 'qabmassnahmen' );
    PERFORM TSystem.trigger__disable( 'wareneingangskontrolle' );
    PERFORM TSystem.trigger__disable( 'mitpln' );

    UPDATE mitpln                    SET mpl_minr           = minr_new   WHERE mpl_minr = minr_old;
    UPDATE maschausf                 SET ma_minr            = minr_new   WHERE ma_minr = minr_old;
    UPDATE stundauszahl              SET sa_minr            = minr_new   WHERE sa_minr = minr_old;
    UPDATE ExportLohn                SET el_minr            = minr_new   WHERE el_minr = minr_old;
    UPDATE bdepftbe                  SET bdfb_decider       = minr_new   WHERE bdfb_decider = minr_old;
    UPDATE personal                  SET pers_vertrauensp   = minr_new   WHERE pers_vertrauensp = minr_old;
    UPDATE personal                  SET pers_vertretung1   = minr_new   WHERE pers_vertretung1 = minr_old;
    UPDATE personal                  SET pers_vertretung2   = minr_new   WHERE pers_vertretung2 = minr_old;
    UPDATE qabmassnahmen             SET qmas_bearbeiter    = minr_new   WHERE qmas_bearbeiter = minr_old;
    UPDATE wareneingangskontrolle    SET wek_pruefer        = minr_new   WHERE wek_pruefer = minr_old;

    UPDATE belegdokument            SET beld_apint          = db_usename_new WHERE beld_apint = db_usename_old;

    UPDATE ldsdokdokutxt            SET ltd_apint2          = minr_new   WHERE ltd_apint2 = minr_old;  -- ltd_apint2  VARCHAR(50),
    IF NOT MiNrZusammenFuehren THEN
     UPDATE llv                SET ll_minr     = minr_new   WHERE ll_minr = minr_old;
    END IF;
    UPDATE bdep_log            SET bdl_minr     = minr_new   WHERE bdl_minr = minr_old;

    PERFORM TSystem.trigger__enable( 'maschausf' );
    PERFORM TSystem.trigger__enable( 'stundauszahl' );
    PERFORM TSystem.trigger__enable( 'ExportLohn' );
    PERFORM TSystem.trigger__enable( 'bdep_log' );
    PERFORM TSystem.trigger__enable( 'bdepftbe' );
    PERFORM TSystem.trigger__enable( 'personal' );
    PERFORM TSystem.trigger__enable( 'qabmassnahmen' );
    PERFORM TSystem.trigger__enable( 'wareneingangskontrolle' );
    PERFORM TSystem.trigger__enable( 'mitpln' );

    IF MiNrZusammenFuehren THEN
     --- DELETE FROM llv WHERE ll_minr = minr_old;
     --- DELETE FROM adk WHERE ad_krz = ll_adkrz_old;
    END IF;

    RESET SESSION AUTHORIZATION;

    RETURN;
  END $$ LANGUAGE plpgsql;
--

-- Funktion zur Berechnung der gerundenten Raucherpausenzeit. Die Rundungsgenauigkeit wird über das Setting 'bde_raucherpause_rndngsgngkt' ermittelt. #9691
CREATE OR REPLACE FUNCTION tpersonal.llv__bdpab__rpause__round(
      minr            integer,
      anf_datum       date,
      end_datum       date,
      nur_unverbuchte boolean = false,
      _forcesum       boolean = false
  ) RETURNS numeric AS $$
  DECLARE
      -- Der Divisor bestimmt die Rundungsgenauigkeit. Muss in Minuten angg. werden.
      _divisor numeric := TSystem.Settings__Get( 'bde_rpause_rundung_faktor' );
      _id_raucherpause CONSTANT int := 103;
  BEGIN

      IF
            minr IS NULL
         OR anf_datum IS NULL
         OR end_datum IS NULL
      THEN

          -- Eingabewerte für Mitarbeiter bzw. Datum sind null.
          RAISE EXCEPTION '%', lang_text(32638);
      END IF;

      IF anf_datum > end_datum THEN
          -- Bis-Datum ist kleiner als das Von-Datum. Bitte prüfen.
          RAISE EXCEPTION '%', lang_text(32639);
      END IF;

      IF TSystem.Settings__GetBool('bdep__saldo__ohne__raucherpause', false) AND _forcesum IS false THEN
        RETURN 0; -- unklar: wenn true, werden die Raucherpausen bereits im bd_saldo abgezogen. Was passiert dann in Schnittstelle/Lohnübergabe.... wahrscheinlich
      END IF;

      -- Zielstellung
      --  • Alle Raucherpausen im Zeitraum addieren
      --  • Summe auf Rundungsfaktor aufrunden (niemals ab)
      -- eventuelle Korrektur: round( TWawi.Round_ToLos( _divisor, sum ( minutes ) ), 2 )   (DG 18.06.2020)
      RETURN (
          SELECT
            -- geändert auf timediff Aufruf mit Datum, bei nächtlichen Raucherpausen über Mitternacht notwendig
            -- BEACHTE FUNCTION tpersonal.llv__bdpab__rpause__round. Doppelcode ggfs nochmal kapseln!
             round(
                round(
                  (
                    sum(
                      round( timediff( bdab_anf + bdab_anft, coalesce( bdab_end, bdab_anf ) + bdab_endt ), 2 )
                    )
                  )
                  * ( 60 / _divisor )
                )
                / ( 60 / _divisor ),
              2
            )
          FROM mitpln
            JOIN bdepab ON     mpl_date BETWEEN bdab_anf AND coalesce ( bdab_end, bdab_anf )
                           AND bdab_minr = mpl_minr
                           AND bdab_aus_id = _id_raucherpause
          WHERE mpl_minr = minr
            AND mpl_date BETWEEN anf_datum AND end_datum
            AND mpl_date < current_date
            AND (
                  -- #14769 FUnktionsaufruf aus Monatsabschluss heraus auch mit Einschränkung auf nur unverbuchte
                  -- Datensätze möglich
                     NOT nur_unverbuchte
                  OR (
                          nur_unverbuchte
                      AND NOT mpl_buch
                  )
                )
      );
  END $$ LANGUAGE plpgsql STABLE PARALLEL SAFE;

CREATE OR REPLACE FUNCTION tpersonal.llv__bdpab__rpause__round__by__bdep(
  IN  _bdep           bdep
  )
  RETURNS numeric AS $$
  DECLARE
      -- Der Divisor bestimmt die Rundungsgenauigkeit. Muss in Minuten angg. werden.
      _divisor numeric := TSystem.Settings__Get( 'bde_rpause_rundung_faktor' );
      _id_raucherpause CONSTANT int := 103;
  BEGIN
      -- Zielstellung
      --  • Alle Raucherpausen im Zeitraum addieren
      --  • Summe auf Rundungsfaktor aufrunden (niemals ab)
      -- eventuelle Korrektur: round( TWawi.Round_ToLos( _divisor, sum ( minutes ) ), 2 )   (DG 18.06.2020)
      RETURN (
          SELECT
            -- geändert auf timediff Aufruf mit Datum, bei nächtlichen Raucherpausen über Mitternacht notwendig
            -- BEACHTE FUNCTION tpersonal.llv__bdpab__rpause__round. Doppelcode ggfs nochmal kapseln!
             round(
                round(
                  (
                    sum(
                      round( timediff( bdab_anf + bdab_anft, bdab_end + bdab_endt ), 2 )
                    )
                  )
                  * ( 60 / _divisor )
                )
                / ( 60 / _divisor ),
              2
            )
           FROM bdepab
          WHERE bdab_minr = _bdep.bd_minr
            AND bdab_end IS NOT null
            AND bdab_aus_id = _id_raucherpause
            AND bdab_anf + bdab_anft BETWEEN _bdep.bd_anf AND coalesce(_bdep.bd_end, now())
      );
  END $$ LANGUAGE plpgsql STABLE PARALLEL SAFE STRICT;

--

-- 14030 Raucherpausenermittlung im Rahmen der Lohnarten
-- vorerst neue Funktionalität nur bei Rundungsfaktor 1
-- Zeitraumumfassend möglich (Monatsabschluss)
-- Tagesaktuelle Abfrage möglich
-- lohnarten-spezifische Abfrage mit Zeiten möglich
CREATE OR REPLACE FUNCTION TPersonal.bdep__raucherpausen_summe__get(
      minr      integer,
      begindate date,
      enddate   date = null,
      begintime time = null,
      endtime   time = null
  ) RETURNS numeric AS $$
  DECLARE
      _rundungsfaktor integer := TSystem.Settings__GetInteger('bde_rpause_rundung_faktor');
      _enddate        date;
      _rpause_sum     numeric(12,4);
  BEGIN

    -- Aufruf der Funktion entweder nur mit Datum(s)feld(ern) und Zeitstempeln leer oder aber mit beiden Zeitstempeln gefüllt
    IF
            (     begintime IS NULL
              AND endtime IS NOT NULL
        )
        OR  (     begintime IS NOT NULL
              AND endtime IS NULL
        )
    THEN
        RAISE EXCEPTION 'xtt32628 - Funktionsaufruf TPersonal.bdep__raucherpausen_summe__get für Mitarbeiter % ', minr;
    END IF;

    IF _rundungsfaktor <> 1 THEN
        -- Aufruf der bisherigen Rundungsfunktion
        -- da aktuell kein einfaches und schnelles umbauen hinsichtlich der Rundungen möglich ;)
        -- 02.11.22 Abstimmung DS,MF zur Änderung #18941: die Einstiegsbedingung ist falsch und müsste geändert werden, da die Rundung auf 1 Minute ausser Kraft gesetzt wurde
          -- aus Zeitgründen und Mangels aktueller Notwendigkeit wurde auf eine Korrektur verzichtet
          -- Info: der Lohnexport in Bezug auf Raucherpausen ist nur richtig, wenn der Rundungsfaktor auf 1 steht, eine Rundung auf Tagesebene ist nicht möglich
        RETURN coalesce( tpersonal.llv__bdpab__rpause__round( minr, begindate, coalesce( enddate, begindate ) ), 0 );
    ELSE
        -- eigentliche neue Funktionalität zur Ermittlung der Raucherpausen
        SELECT
          -- vorherige Rundung auf Minute wurde entfernt #18941, da sonst bei Summe von nur 1 Raucherpause eine zu hohe Abweichung im Vergleich zu tpersonal.llv__bdpab__rpause__round() entstünde (monatsweise Rundung)
          sum(
              timediff(
                bdab_anf + bdab_anft,
                bdab_end + bdab_endt
              )
          )
        INTO _rpause_sum
        FROM bdepab
        WHERE
              bdab_minr = minr
          -- Nur Raucherpausen
          AND bdab_aus_id = 103
          -- notwendig!! Ohne Stempeldaten leere Range!
          -- die Prüfung auf den individuellen Werktag funktioniert nicht bei Verwendung von begintime und endtime..denn dann wird bei Nachtschicht-Pausen nach 0 Uhr der Tag nicht mehr getroffen
          AND ( CASE WHEN begintime IS NULL THEN
                     EXISTS ( SELECT 1 FROM bdep WHERE bd_individwt_mpl_date BETWEEN begindate AND coalesce( enddate, begindate ) )
                ELSE true
                END
              )
          AND (
                -- Möglichkeit 1) Zeitstempel nicht angegeben -> Range über zugehörige Stempelungen ermitteln
                (
                      begintime IS NULL
                  AND
                      -- <@: element is contained by
                      ( bdab_anf || ' ' ||  bdab_anft )::timestamp
                      <@
                      (
                          -- Ergänzung '[]' notwendig um Grenzen mit einzuschließen
                          SELECT tsrange( min(bd_anf), max( coalesce( bd_end, currenttime() ) ), '[]' )
                          FROM bdep
                          WHERE bd_individwt_mpl_date BETWEEN begindate AND coalesce( enddate, begindate )
                            AND bd_minr = minr
                      )
                )

                -- Möglichkeit 2) Angegebene Zeitstempel
                -- Abwesenheit muss zwischen diesen liegen
                OR
                (
                      begintime IS NOT NULL
                  AND bdab_anf +  bdab_anft BETWEEN ( begindate + begintime ) AND  ( enddate + endtime )
                )
          )
        ;

        _rpause_sum := round( coalesce( _rpause_sum, 0 ), 2 );

        RETURN _rpause_sum;

    END IF;

  END $$ LANGUAGE plpgsql;
--

-- Stundenkonto bis zu bestimmten Datum bzw. bis heute
-- verbuchte (llv) und unverbuchte (mitpln) Stunden
CREATE OR REPLACE FUNCTION TPersonal.llv__mitpln__ll_stuko__stundenkonto__calc(
    IN _minr                integer,
    IN _ende_datum          date = current_date,
    IN _raucherpause_abzug  bool = true
    )
    RETURNS numeric
    AS $$

      SELECT
             -- bereits verbuchtes Stundenkonto
               coalesce( ll_stuko_buch, 0 )

             -- plus unverbuchte Stunden laut Präsenzzeiten
             + coalesce(
                   (
                       SELECT
                           sum(
                               -- Rundung pro Tag bevor es summiert wird
                               round(
                                     coalesce( mpl_saldo,   0 )
                                   - coalesce( mpl_min,     0 )
                                   + coalesce( mpl_absaldo, 0 )

                                   -- auf 2 NK runden
                                   , 2
                               )
                           )

                       FROM mitpln
                       -- Berücksichtigung des Vortags bei current_date, um mögliche Konflikte wegen Tag abgeschlossen oder nicht
                       -- letzte Fehlerbehebung durch #20210
                       WHERE mpl_minr = ll_minr
                         AND NOT mpl_buch
                         AND (
                                    _ende_datum IS NULL
                                OR  (mpl_date <= _ende_datum
                                     AND mpl_date < current_date
                                    )
                             )
                         AND ( -- Performance: historische Daten ausfiltern
                                    ( ll_endd IS NULL )
                                OR  ( ll_endd > _ende_datum )
                             )
                              --hier könnt man noch einbauen mpl_date anfang nicht merh als 1 Jahr
                   )

                   , 0  -- Fallback
               )
             - CASE WHEN _raucherpause_abzug
                 -- andernfalls sind die Raucherpausen zu diesem Zeitpunkt bereits verbucht und müssen nicht berechnet werden
                 AND _ende_datum >= coalesce(rp_anf_date, _ende_datum) -- fallback, wenn kein monatsabschluss
                    THEN -- hinweis: TPersonal.bde__status__get verwendet auch dise funktion, weißt die raucherpause aber extra aus daher optional hier
                         -- Anf-Date kann auf kleinstes nicht verbuchtes mitpln-date gesetzt werden, da Parameter 'nur unverbuchte' gesetzt ist
                         coalesce(tpersonal.llv__bdpab__rpause__round( _minr, coalesce(rp_anf_date, '-Infinity'), coalesce(_ende_datum, current_date), true), 0)
                    ELSE
                         0
               END
        FROM llv
            LEFT JOIN LATERAL (   -- Performance::dieses Datum wird benötigt, damit der Aufruf llv__bdpab__rpause__round sinnvoll eingeschränkt werden kann
                                  SELECT
                                    --Datum der letzten Verbuchung + 1 Tag >> entspricht dem ersten Tag, der für Berechnung relevant ist
                                    (yearmonth_to_date( max(llsh_monthyear)::INTEGER,true)+ interval '1 day')::DATE AS rp_anf_date
                                  FROM llv_stuko_history
                                  WHERE llsh_ll_minr=ll_minr
                              ) AS stuko_history ON true
       WHERE ll_minr = _minr

    $$ LANGUAGE sql STABLE;
--

-- Berechnung des Saldos (+/-) im Vergleich zu den Sollstunden im Zeitraum
-- keine Berücksichtigung des verbucht-Status ~ dazu dient TPersonal.llv__mitpln__ll_stuko__stundenkonto__calc
CREATE OR REPLACE FUNCTION TPersonal.llv__mitpln__stundenkonto_saldo__calc(
    IN _minr                integer,
    IN _date_start          date,
    IN _date_end            date = current_date,
    IN _raucherpause_abzug  bool = true
    )
    RETURNS numeric
    AS $$

      SELECT
          -- Stunden laut Präsenzzeiten
          coalesce(
                       sum(
                             coalesce( mpl_saldo,   0 )
                             - coalesce( mpl_min,     0 )
                             + coalesce( mpl_absaldo, 0 )
                          )

                       , 0
          )
          - CASE WHEN _raucherpause_abzug THEN
                         coalesce(tpersonal.llv__bdpab__rpause__round( _minr, _date_start, coalesce(_date_end, current_date), false), 0)
                 ELSE
                      0
                 END
      FROM mitpln
      WHERE
          mpl_minr = _minr
          AND mpl_date BETWEEN _date_start AND _date_end

    $$ LANGUAGE sql STABLE;
--

-- Erstellt Tagesplaneinträge für Mitarbeiter von Heute bis Freitag (Ticket: 5536)
 CREATE OR REPLACE FUNCTION tpersonal.tplan_create_mofr_fromf2(IN src_table VARCHAR, IN src_id INTEGER) RETURNS VOID AS $$
  DECLARE
    tdate DATE;
    c DATE;
    tPlan VARCHAR ARRAY[5]; --[1..5] -> [Mo..Fr]
    pos INTEGER;
    minr INTEGER;
  BEGIN
    IF src_table='llv' THEN
        SELECT ARRAY[ll_standplan_mo, ll_standplan_di, ll_standplan_mi, ll_standplan_do, ll_standplan_fr, ll_standplan_sa, ll_standplan_so]
        FROM llv
        WHERE ll_minr=src_id
        INTO tPlan;

        minr:=src_id;
    ELSEIF src_table='llv_autoplan' THEN
        SELECT llpl_minr, ARRAY[llpl_tpl_name, llpl_tpl_name, llpl_tpl_name, llpl_tpl_name, llpl_tpl_name_fr]
        FROM llv_autoplan
        WHERE llpl_id=src_id
        INTO minr, tPlan;
    END IF;

    c:=current_date;  -- + INTERVAL '1 day'; -- zum Testen
    FOR tdate IN SELECT * FROM GENERATE_SERIES(c, c+5-EXTRACT(isodow FROM c)::INTEGER, '1 day') LOOP -- heute bis Freitag
        pos:=EXTRACT(isodow FROM tdate)::INTEGER;

        IF (tPlan[pos] IS NOT NULL) AND NOT EXISTS(SELECT true FROM feiertag WHERE ft_date=tdate AND NOT ft_urlaub) THEN -- außer echte Feiertage, die kein Betriebsurlaub sind
            -- Update, wenn schon was eingetragen ist
            UPDATE mitpln SET
              mpl_tpl_name=tPlan[pos]
            WHERE mpl_minr=minr
              AND mpl_date=tdate;

            IF NOT FOUND THEN
                -- sonst anlegen
                INSERT INTO mitpln (mpl_minr, mpl_tpl_name, mpl_date)
                VALUES (minr, tPlan[pos], tdate);
            END IF;
        END IF;
    END LOOP;
  END $$ LANGUAGE plpgsql;
--

--
 CREATE OR REPLACE FUNCTION IsPresentMitNr(INTEGER) RETURNS INTEGER AS $$
  DECLARE minr ALIAS FOR $1;
  BEGIN
   RETURN bd_minr FROM bdep WHERE bd_end IS NULL AND bd_minr=minr;
  END $$ LANGUAGE plpgsql;
--

-- Zur BDE durch RFID-Geräte
CREATE OR REPLACE FUNCTION IsPresentRFID(
    IN inRFID VARCHAR,
    OUT isPresent     BOOLEAN,  OUT isRaucherPause  BOOLEAN,  OUT errorMsg  VARCHAR,
    OUT lastStampMsg  VARCHAR,  OUT pauseStampMsg   VARCHAR,
    OUT lastWorkerMsg VARCHAR,  OUT TagSaldoMsg     VARCHAR
  ) RETURNS RECORD AS $$
  DECLARE
    minr        INTEGER;
    saldo       VARCHAR;
    diff        VARCHAR;
    ang_seit    VARCHAR;
    isLast      BOOLEAN;
    timesum     VARCHAR;
    curDate     DATE;
    cur_ind_wt  DATE;
  BEGIN
    curDate         := current_date;
    isPresent       := false;
    isRaucherPause  := false;
    inRFID          := ltrim(inRFID, '0');

    -- Anzahl der Verwendungen der RFID
    CASE ( SELECT count(ll_minr) FROM llv WHERE TSystem.ENUM_GetValue(ll_rfid, inRFID) )

    -- kein MA hat die RFID
    WHEN 0 THEN
        errorMsg := lang_text(16362); -- Fehler: RFID nicht vergeben

    -- RFID ist eindeutig
    WHEN 1 THEN
        -- Mitarbeiter ist ausgeschieden
        IF EXISTS(SELECT true FROM llv WHERE TSystem.ENUM_GetValue(ll_rfid, inRFID) AND ll_endd <= curDate) THEN
            errorMsg := lang_text(16361); -- Fehler: Mitarbeiter ist ausgeschieden

        -- Mitarbeiter ist nicht ausgeschieden. Ergebnisse können berechnet werden
        ELSE
            minr        := ll_minr FROM llv WHERE TSystem.ENUM_GetValue(ll_rfid, inRFID);
            isPresent   := EXISTS(SELECT true FROM bdep WHERE bd_minr = minr AND bd_end IS NULL);
            -- aktueller individueller Werktag anhand Daten
            cur_ind_wt  := TPersonal.bdep__individwt__get_date__by_bd_anf_minr(currenttime(), minr)::DATE;

            -- nicht beendete Raucherpausenstempelung von heute vorhanden
            isRaucherPause :=
                EXISTS(
                  SELECT true
                  FROM bdepab
                  WHERE bdab_minr = minr
                    AND bdab_endt IS NULL
                    AND bdab_anf = curDate -- funktioniert nicht über Mitternacht
                    AND bdab_aus_id = 103
            );

            -- Letzte Eintrag, #9606
            lastStampMsg := (
                SELECT to_char(COALESCE(bd_end_rund, bd_anf_rund), 'DD. Mon HH24:MI')
                FROM bdep
                WHERE bd_minr = minr
                ORDER BY bd_anf DESC
                LIMIT 1
            );

            -- pause seit 'pauseStampStr', #9606
            pauseStampMsg := (
                SELECT to_char(bdab_anft, 'HH24:MI')
                FROM bdepab
                WHERE bdab_minr = minr
                  AND bdab_aus_id = 103
                  AND bdab_endt IS NULL
                  AND bdab_anf = curDate -- funktioniert nicht über Mitternacht
                ORDER BY bdab_anf DESC, bdab_anft DESC
                LIMIT 1
            );

            -- Erweiterungen der Info-Anzeige und Tagesarbeitszeit beim Abstempeln #9811
            -- Der Mitarbeiter ist der letzte, #9811
            isLast :=
                NOT EXISTS(
                  SELECT true
                  FROM bdep
                  WHERE bd_minr <> minr
                    AND bd_end IS NULL
                    -- AND bd_anf::DATE = curDate
            );

            IF isLast THEN
                lastWorkerMsg := lang_text(26293); -- Letzter Mitarbeiter
            END IF;

            -- angestempelt seit
            ang_seit := (
                SELECT to_char(bd_anf_rund, 'HH24:MI')
                FROM bdep
                WHERE bd_end IS NULL
                  AND bd_individwt_mpl_date = cur_ind_wt
                  AND bd_minr = minr
                ORDER BY bd_anf DESC
                LIMIT 1
            );

            -- #10624
            timesum :=
                to_char(
                  (   SELECT
                        SUM(
                          CASE
                            WHEN (bd_end IS NOT NULL OR bd_saldo <> 0) THEN
                                bd_saldo
                            ELSE
                                timediff(bd_anf, currenttime())
                          END
                        )::NUMERIC(12,2)
                        || 'h' AS bd_timesum
                      FROM bdep
                      WHERE (    bd_end IS NULL
                              OR bd_individwt_mpl_date = cur_ind_wt
                            )
                        AND bd_minr = minr
                  )::INTERVAL
                  , 'HH24:MI' -- to_char
            );

            -- #9811
            IF timesum IS NOT NULL THEN
                IF ang_seit IS NULL THEN
                    TagSaldoMsg := timesum;
                ELSE
                    TagSaldoMsg := ang_seit || ' (' || timesum || ')';
                END IF;
            ELSE
                TagSaldoMsg := TagSaldoMsg || 'leer';
            END IF;
        END IF;

    -- mehrere MA haben die RFID
    ELSE
        errorMsg := lang_text(16363); -- Fehler: RFID mehrfach vergeben
    END CASE;

    RETURN;
  END $$ LANGUAGE plpgsql STABLE STRICT;
--


/* Kennzahlen Abwesenheiten 6883 */

  -- Grundfunktion für die Summierung von Abwesenheitszeiten auf Basis mitpln
  CREATE OR REPLACE FUNCTION tpersonal.llv__mitpln__abwsum__by__minr__date__get(
    IN _minr integer,
    IN _abw_id integer,
    IN _futuredate boolean,
    IN _date date,
    IN _NurWochentage boolean DEFAULT False
    )
    RETURNS NUMERIC
    AS $$
      SELECT
        COALESCE(
          CASE WHEN _futuredate THEN
            (   -- futuredate Bezeichnung bezieht sich auf Testfähigkeit, in Realität ist mitpln-Eintrag in der Zukunft nicht da
                  -- bezieht sich explizit auf die nicht verbuchten Abwesenheiten
                  -- Parametername wegen Übersichtlichkeit aus tpersonal.llv__urlaub__rest__by__futuredate__get übernommen
                -- EndeDatum DATE DEFAULT NULL
                SELECT COUNT(1)
                  FROM mitpln
                  JOIN bdepab ON bdab_minr = mpl_minr AND mpl_date BETWEEN bdab_anf AND bdab_end AND bdab_aus_id = _abw_id AND NOT bdab_buch
                    -- Hole den Tag zur Abwesenheit (vor count) und filtere es unten im WHERE raus. Somit hat count letztlich nur noch die Arbeitstage 1..5
                  LEFT JOIN LATERAL day_of_week(mpl_date) ON _NurWochentage
                 WHERE mpl_minr = ll_minr
                   AND mpl_feiertag IS NULL
                    -- Beachte Lateral JOIN wo die Tage extrahiert werden um diese hier zu filtern
                    -- NOT nur Wochentage heißt alle Tage.
                   AND ( NOT _NurWochentage OR ( day_of_week BETWEEN 1 AND 5 ) )
                   AND ( _date IS NULL OR ( mpl_date <= _date ) ) -- nach #20357
            )
          ELSE
            (   -- im laufenden Jahr bis heute bzw _date
                -- IN _date date DEFAULT current_date
                SELECT COUNT(1)
                  FROM mitpln
                  JOIN bdepab ON bdab_minr = mpl_minr AND mpl_date BETWEEN bdab_anf AND bdab_end AND bdab_aus_id = _abw_id
                    -- Hole den Tag zur Abwesenheit (vor count) und filtere es unten im WHERE raus. Somit hat count letztlich nur noch die Arbeitstage 1..5
                  LEFT JOIN LATERAL day_of_week(mpl_date) ON _NurWochentage
                 WHERE mpl_minr = ll_minr
                   AND mpl_feiertag IS NULL
                    -- Beachte Lateral JOIN wo die Tage extrahiert werden um diese hier zu filtern
                    -- NOT nur Wochentage heißt alle Tage.
                   AND ( NOT _NurWochentage OR ( day_of_week BETWEEN 1 AND 5 ) )
                   AND bdab_anf >= date_trunc( 'year', COALESCE( _date, current_date) )
            )
          END, 0)
       FROM llv
      WHERE ll_minr = _minr;
    --END
    $$ LANGUAGE sql; --plpgsql STABLE
--

-- #14772 Urlaubsbuchungen im Zeitraum
CREATE OR REPLACE FUNCTION TPersonal.bdep__urlaub__by_monat__sum__get(
      _minr            integer,
      _datum_von       date,
      _datum_bis       date,
      _nur_unverbuchte boolean = true
  ) RETURNS numeric AS $$
  DECLARE
      _sum             numeric := 0;
      _urlaub_nur_tage boolean;

      -- Urlaubstag und halber Urlaubtstag
      _urlaubstag_id        CONSTANT int := 1;
      _halber_urlaubstag_id CONSTANT int := 100;
  BEGIN

      IF
            _minr IS NULL
         OR _datum_von IS NULL
         OR _datum_bis IS NULL
      THEN
          -- Eingabewerte für Mitarbeiter bzw. Datum sind null.
          RAISE EXCEPTION '%', lang_text( 32638 );
      END IF;

      IF _datum_von > _datum_bis THEN
          -- Bis-Datum ist kleiner als das Von-Datum. Bitte prüfen.
          RAISE EXCEPTION '%', lang_text( 32639 );
      END IF;

      _urlaub_nur_tage := ll_urlaub_days
                         FROM llv
                         WHERE ll_minr = _minr;

      IF _urlaub_nur_tage THEN

          -- Urlaub ganzer Tag
          _sum := TPersonal.bdep__urlaubstage__by_minr_zeitraum__get( _minr, _datum_von, _datum_bis, _urlaubstag_id, _nur_unverbuchte );

          -- Urlaub halbe Tage
          _sum := _sum + TPersonal.bdep__urlaubstage__by_minr_zeitraum__get( _minr, _datum_von, _datum_bis, _halber_urlaubstag_id, _nur_unverbuchte );

      ELSE

          -- Urlaub Stundenweise
          _sum :=      sum( bdab_stu )
                 FROM bdepabsum
                 JOIN mitpln    ON  mitpln.mpl_minr = bdepabsum.mpl_minr
                                AND mitpln.mpl_date = bdepabsum.mpl_date
                 WHERE bdepabsum.mpl_minr=_minr
                   AND (
                         ( _nur_unverbuchte AND NOT mpl_buch )
                          OR
                          NOT _nur_unverbuchte
                       )
                   AND mitpln.mpl_date BETWEEN _datum_von AND _datum_bis
                   AND ab_id IN ( _urlaubstag_id, _halber_urlaubstag_id );
      END IF;

      RETURN coalesce( _sum, 0 );

  END $$ LANGUAGE plpgsql;

-- Urlaub/Krank genommen im laufenden Jahr (ohne Berücksichtigung bdab_buch) #6883 #20357
-- SELECT tsystem.function__drop_by_regex( 'llv__urlaub__or__krank__by__currentyear__get', _commit => true);
CREATE OR REPLACE FUNCTION tpersonal.llv__urlaub__or__krank__by__currentyear__get(
    IN _minr integer,
    IN _NurKranktage  boolean DEFAULT False,
    IN _NurWochentage boolean DEFAULT False,
    IN _date date DEFAULT current_date
    )
    RETURNS NUMERIC
    AS $$
      SELECT
        COALESCE(
            CASE WHEN ll_urlaub_days AND NOT _NurKranktage THEN -- keine Kranktage zählen
              (
                --Urlaub Tage
                tpersonal.llv__mitpln__abwsum__by__minr__date__get(
                    _minr,
                    _abw_id => 1,
                    _futuredate => false,
                    _date => _date,
                    _NurWochentage => _NurWochentage
                )
              )
              +
              (
                --Urlaub Tage (1/2)
                tpersonal.llv__mitpln__abwsum__by__minr__date__get(
                    _minr,
                    _abw_id => 100,
                    _futuredate => false,
                    _date => _date,
                    _NurWochentage => _NurWochentage
                ) * 0.5
              )
              +
              (
                --virtuellen Tag addieren, für Ausgabe am heutigen Tag #20357
                SELECT
                  COALESCE(
                    (SELECT
                     CASE
                       WHEN bdab_aus_id = 100 THEN 0.5
                       ELSE 1
                     END
                    FROM
                      bdepab
                      LEFT JOIN LATERAL day_of_week(current_date) ON _NurWochentage
                    WHERE
                      bdab_minr = ll_minr
                      AND current_date BETWEEN bdab_anf AND bdab_end
                      AND (bdab_aus_id = 1 OR bdab_aus_id = 100)
                      AND (NOT _NurWochentage OR (day_of_week BETWEEN 1 AND 5))
                      --AND NOT _NurKranktage
                      ),
                    0
                  )
              )
            WHEN ll_urlaub_days AND _NurKranktage THEN -- Kranktage-Ausweisung
              (
                -- Kranktage
                tpersonal.llv__mitpln__abwsum__by__minr__date__get(
                    _minr,
                    _abw_id => 2,
                    _futuredate => false,
                    _date => _date,
                    _NurWochentage => _NurWochentage
                )
              )
              +
              (
                --virtuellen Tag addieren, für Ausgabe am heutigen Tag #20357
                SELECT
                  COALESCE(
                    (SELECT 1
                    FROM
                      bdepab
                      LEFT JOIN LATERAL day_of_week(current_date) ON _NurWochentage
                    WHERE
                      bdab_minr = ll_minr
                      AND current_date BETWEEN bdab_anf AND bdab_end
                      AND (bdab_aus_id = 2)
                      AND (NOT _NurWochentage OR (day_of_week BETWEEN 1 AND 5))
                      ),
                    0
                  )
              )
            ELSE
              (
                -- Urlaub Stunden
                -- war/ist bisher nicht fertig implementiert,
                -- todo analog oben in Grundfunktion tpersonal.llv__mitpln__abwsum__by__minr__date__get mit neuer Option Stunden übernehmen und hier else auflösen
                SELECT SUM(mpl_absaldo)
                  FROM mitpln
                  JOIN bdepab ON bdab_minr = mpl_minr AND mpl_date BETWEEN bdab_anf AND bdab_end AND bdab_aus_id = 1
                  LEFT JOIN LATERAL day_of_week(mpl_date) ON _NurWochentage
                 WHERE mpl_minr = ll_minr AND mpl_feiertag IS NULL  -- ACHTUNG: Stunden an Feiertagen wurden früher mitgezählt, da "AND mpl_feiertag IS NULL" fehlte.
                   AND (NOT _NurWochentage OR (day_of_week BETWEEN 1 AND 5))
                   AND bdab_anf >= date_trunc('year', _date)
              )
            END, 0)
       FROM llv
      WHERE ll_minr = _minr;
    $$ LANGUAGE sql;
--

CREATE OR REPLACE FUNCTION tpersonal.llv__urlaub__by__currentyear__get(
    IN _minr integer,
       -- Urlaub wir per Default an Wochenenden NICHT eingetragen!???
    IN _NurWochentage boolean DEFAULT False,
    IN _date date DEFAULT current_date
    )
    RETURNS numeric
    AS $$
        SELECT tpersonal.llv__urlaub__or__krank__by__currentyear__get(_minr, _NurKranktage => false, _NurWochentage => _NurWochentage, _date => _date);
    $$ LANGUAGE sql STABLE;

CREATE OR REPLACE FUNCTION tpersonal.llv__krank__by__currentyear__get(
    IN _minr integer,
       -- Urlaub wir per Default an Wochenenden NICHT eingetragen!???
    IN _NurWochentage boolean DEFAULT False,
    IN _date date DEFAULT current_date
    )
    RETURNS numeric
    AS $$
        SELECT tpersonal.llv__urlaub__or__krank__by__currentyear__get(_minr, _NurKranktage => true, _NurWochentage => _NurWochentage, _date => _date);
    $$ LANGUAGE sql STABLE;


-- #22296 Berechnung
-- Es gibt ältere Monatsabschlüsse im Bereich des Eingabedatum
-- Der zum Eingabedatum gültige Urlaubsanspruch berechnet sich über den verbuchten Wert im letzten Monat vor dem Eingabedatum abzüglich der Urlaubstage vom 01. des Folgemonats bis zum Eingabedatum
-- Urlaubsanspruch zum 10.09.2020 ist Monatsaschlusswert 202008 abzuüglich Tage im Zeitraum 01.09.2020 bis 10.09.2020
CREATE OR REPLACE FUNCTION tpersonal.llv__urlaub__rest__by__pastdate__get(
  _minr INTEGER,
  _date DATE,
  _NurWochentage BOOLEAN DEFAULT False --aktuell nicht verwendet, muss nachgesetzt werden
  )
  RETURNS NUMERIC
  AS $$

    WITH last_record AS (
        SELECT llsh_urlaub, yearmonth_dec_to_date(llsh_monthyear::INTEGER, true) AS last_month_date
        FROM llv_stuko_history
        WHERE llsh_ll_minr = _minr
          AND yearmonth_dec_to_date(llsh_monthyear::INTEGER, true) < _date
        ORDER BY llsh_monthyear DESC
        LIMIT 1
    )
    -- vom verbuchter Wert im letzten Monat vor dem Eingabedatum ziehen wir die
    -- Urlaubstage vom 01. des Folgemonats bis zum Eingabedatum ab
    SELECT
        ROUND(
          COALESCE(
              ( SELECT
                  COALESCE(lr.llsh_urlaub, 0)  -- Falls kein Wert gefunden wird, 0 als Standard
                  - COALESCE(TPersonal.bdep__urlaub__by_monat__sum__get( _minr,
                                                                         COALESCE(lr.last_month_date + 1, _date), -- vom 01. des Folgemonats
                                                                         _date,
                                                                         false) -- nur unverbuchte false kann immer angenommen werden
                                                                         , 0)
                    -- Hinweis zum Zwischenstand: bdep__urlaub__by_monat__sum__get kann aktuell nur zwischen verbucht und unverbucht unterscheiden
                    -- es ist erweiterte Funktion mit Parameter nur _NurWochentage erforderlich
              FROM last_record lr
              ), 0  -- Falls das gesamte SELECT NULL ergibt, wird 0 zurückgegeben
        ), 1  -- auf eine Nachkommastelle runden
      );
  $$ LANGUAGE sql;


-- Berechnung 'verbuchter Resturlaub' abzüglich 'Urlaub genommen' (bis zu optionalem Datum) ~ ohne Urlaub in der Zukunft //entspricht auch TMitNr ll_urlaub_act
CREATE OR REPLACE FUNCTION tpersonal.llv__urlaub__rest__by__futuredate__get(
  minr INTEGER,
  NurWochentage BOOLEAN DEFAULT False,
  EndeDatum DATE DEFAULT NULL
  )
  RETURNS NUMERIC
  AS $$
    SELECT
      COALESCE(ll_urlaub,0) - COALESCE(
        CASE WHEN ll_urlaub_days THEN
          ( -- Urlaub ~ futuredate und nicht verbuchte
            tpersonal.llv__mitpln__abwsum__by__minr__date__get(
                _minr => minr,
                _abw_id => 1,
                _futuredate => true,
                _date => EndeDatum,
                _NurWochentage => NurWochentage
            )
          )
          +
          (
            --Urlaub Tage (1/2)
            tpersonal.llv__mitpln__abwsum__by__minr__date__get(
                _minr => minr,
                _abw_id => 100,
                _futuredate => true,
                _date => EndeDatum,
                _NurWochentage => NurWochentage
            ) * 0.5
          )
          +
          (
            --virtuellen Tag addieren, für Ausgabe am heutigen Tag;
            SELECT
              COALESCE(
                (SELECT
                   CASE
                     WHEN bdab_aus_id = 100 THEN 0.5
                     ELSE 1
                   END
                 FROM
                   bdepab
                   LEFT JOIN LATERAL day_of_week(current_date) ON NurWochentage
                 WHERE
                   bdab_minr = ll_minr
                   AND ( bdab_aus_id = 1 OR bdab_aus_id = 100 )
                   AND NOT bdab_buch
                   AND ( NOT NurWochentage OR (day_of_week BETWEEN 1 AND 5) )
                   -- der Tag ist noch nicht in mitpln vorhanden, dann zählen wir hier den Tag hinzu
                   AND current_date BETWEEN bdab_anf AND bdab_end
                   -- der tag ist in mitplan vorhanden, dann wird es über count mitpln gezählt
                   AND NOT EXISTS(SELECT true FROM mitpln WHERE current_date = mpl_date AND mpl_minr = bdab_minr) -- 20749 CI Testfähigkeit
                ), 0
              )
          )
        ELSE
          (
            -- Urlaub Stunden
            -- war/ist bisher nicht fertig implementiert,
            -- todo analog oben in Grundfunktion tpersonal.llv__mitpln__abwsum__by__minr__date__get mit neuer Option Stunden übernehmen und hier else auflösen
            SELECT SUM(mpl_absaldo)
            FROM mitpln
              JOIN bdepab ON bdab_minr = mpl_minr AND mpl_date BETWEEN bdab_anf AND bdab_end AND bdab_aus_id = 1 AND NOT bdab_buch
              LEFT JOIN LATERAL day_of_week(mpl_date) ON NurWochentage
            WHERE mpl_minr = ll_minr
              AND mpl_feiertag IS NULL  -- ACHTUNG: Stunden an Feiertagen wurden früher mitgezählt, da "AND mpl_feiertag IS NULL" fehlte.
              AND (EndeDatum IS NULL OR (mpl_date <= EndeDatum))
              AND (NOT NurWochentage OR (day_of_week BETWEEN 1 AND 5))
          )
        END, 0)
    FROM llv
    WHERE ll_minr = minr;
  $$ LANGUAGE sql;
--

CREATE OR REPLACE FUNCTION tpersonal.end__of__employees__year__get(
    _minr integer,   -- Mitarbeiternummer
    _date date = current_date
) RETURNS date AS $$

    -- ermittelt den letzten Beschäftigungstag des Jahres eines Mitarbeiters
    -- das Ausscheidungsdatum ist nur im Jahr des InDate relevant Fehler #20609
    SELECT coalesce(
                ( SELECT ll_endd FROM llv WHERE ll_minr = _minr
                    AND ( date_trunc('year', ( _date + interval '1 year' )) ::date  - 1 )
                      = ( date_trunc('year', ( ll_endd + interval '1 year' )) ::date  - 1 )
                ),
                date_trunc('year', ( _date + interval '1 year' )) ::date  - 1 -- wird 31.12.
    );

$$ LANGUAGE sql STABLE;


CREATE OR REPLACE FUNCTION tpersonal.vacation__in__rest__of__employees__year__is(
    _minr integer,   -- Mitarbeiternummer
    _bdab_anf date,  -- Urlaubsbeginn
    _bdab_end date,  -- Urlaubsbeginn
    _date date = current_date
) RETURNS boolean AS $$
    -- prüft, ob ein beantragter Urlaub in das noch verbleibende Jahr fällt,
    -- unter Berücksichtigung des Ausscheidungsdatums des Mitarbeiters

    SELECT daterange( _bdab_anf, _bdab_end, '[]' ) && daterange( least(_date, eoy) , eoy, '[]' )
      FROM tpersonal.end__of__employees__year__get( _minr, _date ) AS eoy

$$ LANGUAGE sql STABLE;


-- >> functions:    FUNCTION tpersonal.llv__urlaub__planbar__by__currentyear__get
--                  FUNCTION tpersonal.llv__urlaub__geplant__beantragt__by__currentyear__get


-- Anwesenheitstage im Zeitraum #7515
CREATE OR REPLACE FUNCTION TPersonal.mitpln__attandance_day__count(
      _minr integer,
      _anf_datum date,
      _end_datum date = current_date
  )
  RETURNS bigint
  AS $$

     SELECT count(*)
       FROM mitpln
      WHERE mpl_minr = _minr
        AND mpl_saldo > 0
        AND mpl_date BETWEEN _anf_datum AND _end_datum;
  $$ LANGUAGE sql STABLE;
--


-- #11462 #11528   Einstellungen > Settings > Modul-Definitionen > BDE > Präsenzzeit > LogoutZeiterfassung nach Anstempeln
CREATE OR REPLACE FUNCTION TPersonal.bde__BDEAutoLogout(BDEUser VARCHAR, DefaultBDELogin BOOL, RFIDLogin BOOL) RETURNS BOOLEAN AS $$
  BEGIN
    -- BDEUser            aktuell geladenes Login
    -- DefaultBDELogin    AutoLogin, siehe Startparameter -BDELogin     https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Konfiguration_(CONFIGSERVERS)
    -- RFIDLogin          wurde durch eine RFID geladen
    RETURN CASE Settings__GetInteger('BDEAutoLogout_Option')
      WHEN 1 THEN current_user = 'BDE'    -- nur BDE-User automatisch abmelden
      WHEN 0 THEN True                    -- immer alle Nutzer abmelden
      ELSE        False
    END;
  END $$ LANGUAGE plpgsql STABLE;
--

-- #12743 Anpassung der Urlaubsplantafel an die Suche für mehrere Abteilungen gleichzeitig
CREATE OR REPLACE FUNCTION TPersonal.abteilung_in_liste(IN abteilung VARCHAR, IN abteilungsliste VARCHAR) RETURNS BOOLEAN AS $$
  DECLARE
    abt VARCHAR;
  BEGIN
  abt:= COALESCE(abteilung, '');
  IF char_length(ltrim(abteilungsliste, ' %')) = 0 OR abteilungsliste IS NULL
    THEN RETURN TRUE;
    ELSE RETURN abt = ANY(string_to_array(abteilungsliste, ','));
  END IF;
END $$ LANGUAGE plpgsql IMMUTABLE;
--


-- Zurücksetzen aller BDE-Daten und Initialisierung mit konkreten Monatsabschluss, siehe #13296
CREATE OR REPLACE FUNCTION TPersonal.llv__bde_reset__stuko_initialization(
    in_ll_minr            INTEGER,
    in_target_month       INTEGER,
    in_target_stuko       NUMERIC,
    in_target_resturlaub  NUMERIC
  ) RETURNS VOID AS $$
  BEGIN
    -- nur Nutzer in Gruppe SYS.Personaldaten dürfen, sonst Abbruch mit Fehler.
    IF NOT EXISTS(SELECT true FROM pg_user WHERE usesysid IN (SELECT unnest(grolist) FROM pg_group WHERE groname = 'SYS.Personaldaten') AND usename = current_user) THEN
        RAISE EXCEPTION '%', lang_text(16776);
    END IF;

    -- nur durchführen, wenn Setting aktiv ist. Setting wird täglich deaktiviert.
    IF NOT Settings__GetBool('BDE.reset.STUKO.initialization.allowed') THEN
        RAISE EXCEPTION '%', lang_text(16777);
    END IF;

    -- Sicherung der Monatsabschlüsse
        CREATE TABLE IF NOT EXISTS z_99_drop.llv_stuko_history_save_13296 (LIKE llv_stuko_history);
        ALTER TABLE z_99_drop.llv_stuko_history_save_13296 ADD COLUMN IF NOT EXISTS save_time_13296 TIMESTAMP(0); -- Zeitpunkt der Sicherung

        INSERT INTO z_99_drop.llv_stuko_history_save_13296
        SELECT *, transaction_timestamp()
        FROM llv_stuko_history
        WHERE llsh_ll_minr    = in_ll_minr
          AND llsh_monthyear  >= in_target_month - 1;
    --

    -- Monatsabschlüsse entfernen für Initialisierung
    -- Achtung! Monate danach werden ggf. wieder geöffnet.
    DELETE FROM llv_stuko_history
    WHERE llsh_monthyear      >= in_target_month - 1
      AND llsh_ll_minr        = in_ll_minr;

    UPDATE bdep SET
      bd_buch = true
    WHERE bd_individwt_mpl_date <= yearmonth_to_date(in_target_month, true)
      AND bd_minr             = in_ll_minr
      AND NOT bd_buch;

    UPDATE bdepab SET
      bdab_buch = true
    WHERE bdab_end            <= yearmonth_to_date(in_target_month, true)
      AND bdab_minr           = in_ll_minr
      AND NOT bdab_buch;

    -- UPDATE bdepabbe SET -- ?

    UPDATE mitpln SET
      mpl_buch = true
    WHERE mpl_date            <= yearmonth_to_date(in_target_month, true)
      AND mpl_minr            = in_ll_minr
      AND NOT mpl_buch;

    UPDATE stundauszahl SET
      sa_buch = true
    WHERE sa_date             <= yearmonth_to_date(in_target_month, true)
      AND sa_minr             = in_ll_minr
      AND NOT sa_buch;

    -- angegebenen Monatsabschluss eintragen
    INSERT INTO llv_stuko_history(llsh_ll_minr, llsh_monthyear,      llsh_stuko,      llsh_urlaub,          llsh_txt)
    VALUES                       (in_ll_minr,   in_target_month - 1, in_target_stuko, in_target_resturlaub, 'BDE reset and STUKO initialization')
    ;

    UPDATE llv SET
      ll_stuko_buch = in_target_stuko,
      ll_urlaub     = in_target_resturlaub
    WHERE ll_minr   = in_ll_minr;

    RETURN;
  END $$ LANGUAGE plpgsql VOLATILE STRICT;
--

CREATE OR REPLACE FUNCTION TPersonal.bdep__anf_end__round__by__tplan(
      _anf_time   timestamp,      -- Anfangszeit der Stempelung
      _end_time   timestamp,      -- Endezeit der Stempelung
      _tpl_name   varchar = null, -- Tagesplan
      OUT _anf_time_rounded timestamp,  -- Anfangszeit gerundet
      OUT _end_time_rounded timestamp   -- Endezeit gerundet
  ) RETURNS record AS $$
  DECLARE
      _tplan_data record;
  BEGIN
    -- Rundung von BDE-Anfangs- und -Endezeit entspr. Tagesplan


    -- Mindestens eine der Zeit muss angg. sein, sonst raus.
    IF _anf_time IS NULL AND _end_time IS NULL THEN   RETURN;   END IF;

    -- notwendige Tagesplan-Daten
    SELECT
      tpl_zeitrund,
      tpl_rbegin,
      tpl_rend
    FROM tplan
    WHERE tpl_name = _tpl_name
    INTO
      _tplan_data -- direkte Zuweisung bei record nicht möglich
    ;

    -- Anfangszeit runden
    _anf_time_rounded :=
        (   TPersonal.date_time__round__by__rule(
                _anf_time,
                _tplan_data.tpl_rbegin,
                _tplan_data.tpl_zeitrund,
                true  -- aufrunden
            )
        )._date_time_rounded  -- Ergebnis
    ;

    -- Endezeit runden
    _end_time_rounded :=
        (   TPersonal.date_time__round__by__rule(
                _end_time,
                _tplan_data.tpl_rend,
                _tplan_data.tpl_zeitrund,
                false -- abrunden
            )
        )._date_time_rounded  -- Ergebnis
    ;


    RETURN;
  END $$ LANGUAGE plpgsql STABLE;
--

-- Funktionen um festzustellen, ob angestempelt ist bzw. wann zuletzt angetempelt war.
  CREATE OR REPLACE FUNCTION tpersonal.bdea__running_active__ba_anf__get(
        _ab_ix      integer,
        _a2_n       integer,
        _mitnr      integer = current_user_minr(),
        OUT ba_anf  timestamp,
        OUT ba_id   integer
    ) RETURNS record AS $$

       SELECT ba_anf, ba_id
         FROM bdea
        WHERE ba_ix   = _ab_ix
          AND ba_op   = _a2_n
          AND ba_minr = _mitnr
          AND ba_efftime IS NULL
        ORDER BY ba_anf DESC
        LIMIT 1;

    $$ LANGUAGE sql;

  CREATE OR REPLACE FUNCTION tpersonal.bdea__running_active__ba_anf__get(
        _ab2        ab2,
        _mitnr      integer = current_user_minr(),
        OUT ba_anf  timestamp,
        OUT ba_id   integer
    ) RETURNS record AS $$

        SELECT tpersonal.bdea__running_active__ba_anf__get(
            _ab2.a2_ab_ix,
            _ab2.a2_n,
            current_user_minr()
        );

    $$ LANGUAGE sql;

  CREATE OR REPLACE FUNCTION tpersonal.bdea__running_last__ba_anfend__get(
        _ab_ix               integer,
        _a2_n                integer,
        _mitnr               integer = current_user_minr(),
        OUT ba_anfend__last  timestamp,
        OUT ba_id            integer
    ) RETURNS record AS $$

        SELECT
            -- wegen Zeitnachtrag auch ba_anf
            coalesce( ba_end, ba_anf ) AS ba_anfend__last,
            ba_id
          FROM bdea
         WHERE ba_ix   = _ab_ix
           AND ba_op   = _a2_n
           AND ba_minr = _mitnr
           AND ba_efftime IS NOT NULL
         ORDER BY ba_anf DESC
         LIMIT 1;

    $$ LANGUAGE sql;

  CREATE OR REPLACE FUNCTION tpersonal.bdea__running_last__ba_anfend__get(
        _ab2                 ab2,
        _mitnr               integer = current_user_minr(),
        OUT ba_anfend__last  timestamp,
        OUT ba_id            integer
    ) RETURNS record AS $$

        SELECT tpersonal.bdea__running_last__ba_anfend__get(
            _ab2.a2_ab_ix,
            _ab2.a2_n,
            current_user_minr()
        );

    $$ LANGUAGE sql;
--

--
CREATE OR REPLACE FUNCTION TPersonal.bdea__auftragszeit_in_zeitraum__ba_ids__get(
      -- Start und Ende des abgefragten Zeitraums
      -- Beachte Berücksichtigung des individ. Werktags der zug. Präsenzzeiten
      _zeitraum_anf date,
      _zeitraum_end date,

      _ab_ix        integer = null, -- ABK
      _minr         integer = null  -- Mitarbeiter
  ) RETURNS SETOF integer AS $$
    -- Ermittlung der Auftragszeiten-IDs im spezifischen Zeitraum, siehe #14979
    -- Optional: Einschränkung auf spezifische ABK und/oder spezifischer MA
    -- Beachte Berücksichtigung vom individ. Werktags der zug. Präsenzzeiten je MA (siehe bdep_minr),
    -- womit Ergebniszeitraum aufgrund evtl. Nachtschicht erweitert wird (Auftragszeiten des folgenden Kalendertags).
  BEGIN

    RETURN QUERY
    SELECT ba_id
      FROM bdea
           -- Hole anhand des individ. Werktags der zug. Präsenzzeiten den evtl. abweichenden Zeitraum für die Auftragszeiten.
           -- Damit werden Auftragszeiten der Nachtschicht (also die des folgenden Kalendertags) automatisch hinzugefügt.
           LEFT JOIN LATERAL (
               SELECT  -- erster Tag der Präsenzzeit im Auftragszeitraum. least: https://redmine.prodat-sql.de/issues/17223?tab=notes#note-24
                       least( min( bd_anf::date ), _zeitraum_anf ) AS min_anf_date,
                       -- letzter Tag der Präsenzzeit im Auftragszeitraum.
                       -- Bei offener Präsenzzeit wird Folgetag angenommen.
                       greatest( max( coalesce( bd_end, bd_anf + '1 day'::interval )::date ), _zeitraum_end ) AS max_end_date
                 FROM bdep
                WHERE  -- je Mitarbeiter
                       bd_minr = ba_minr
                       -- individ. Werktag der Präsenzzeiten des abgefragten Zeitraums
                  AND  bd_individwt_mpl_date BETWEEN _zeitraum_anf AND _zeitraum_end
           ) AS bdep_minr
           ON
             -- Berücksichtigung der Präsenzzeiten nur notwendig bei Angabe von:
                 _zeitraum_anf IS NOT NULL
             AND _zeitraum_end IS NOT NULL
             AND ba_minr IS NOT NULL

     WHERE ( -- spezifische ABK oder alle
                 _ab_ix IS NULL
             OR  ba_ix = _ab_ix
           )

       AND ( -- spezifischer Mitarbeiter oder alle
                 _minr IS NULL
             OR  ba_minr = _minr
           )

       AND ( -- Zeitraum der Auftragszeiten oder offene
               ( -- offene Auftragszeiten
                     ba_end IS NULL
                 AND ba_efftime IS NULL
               )

               -- Auftragszeit innerhalb Zeitraum
                 -- ggf. abweichender Zeitraum aufgrund der Nachtschicht des MA, siehe bdep_minr
               -- Anfang der Auftragszeiten
               OR ba_anf::date BETWEEN coalesce( min_anf_date, _zeitraum_anf )::date AND coalesce( max_end_date, _zeitraum_end )::date
               -- Ende der Auftragszeiten
               OR ba_end::date BETWEEN coalesce( min_anf_date, _zeitraum_anf )::date AND coalesce( max_end_date, _zeitraum_end )::date
           )
     ;
  END
  $$ LANGUAGE plpgsql STABLE PARALLEL SAFE COST 0.01 ROWS 10; -- Hinweis: LANGEUAGE SQL aus irgendeinem Grund wesentlich langsamer. https://redmine.prodat-sql.de/issues/18269
--

--
CREATE OR REPLACE FUNCTION TPersonal.date_time__round__by__rule(
      _date_time    timestamp,        -- ungerundeter Zeitpunkt
      _round_zero   time,             -- Nullpunkt der Rundung (tpl_rbegin, tpl_rend)
      _round_rule   numeric,          -- Vorschrift zur Rundung (tpl_zeitrund), aktuell definiert in Minuten
      _round_up     boolean = false,  -- Richtung der Rundung (Default: abrunden)

      OUT _date_time_rounded    timestamp,  -- gerundeter Zeipunkt
      OUT _date_time_remainder  interval    -- Rest vom Zeitpunkt nach Rundung
  ) RETURNS record AS $$
  DECLARE
      _round_zero_minutes_remainder integer;  -- Rest zum Nullpunkt der Rundung
      _minutes                      integer;
      _minutes_round_factor         integer;  -- ganzzahliger Faktor auf Rundungsvorschrift in Minuten für gerundete Minuten
      _hours                        integer;
  BEGIN
    -- Rundung eines Zeitstempels (Datum mit Zeit) anhand Nullpunkt, Vorschrift (in Minuten) und Richtung


    -- ohne Zeit raus.
    IF _date_time IS NULL THEN    RETURN;   END IF;

    -- ohne Tagesplan-Vorschriften bloße "Rundung" auf Minuten
    IF _round_rule IS NULL THEN
        _date_time_rounded := date_trunc( 'minute', _date_time );

    -- anhand Tagesplan-Vorschriften
    ELSE  -- _round_rule IS NOT NULL
        -- mindestens auf 1 Minute runden
        _round_rule := greatest( _round_rule, 1 );

        -- Wir schauen zuerst, ob die Anfangszeit auch in die Rundung passt, ansonsten müssen wir eine Differenz einbeziehen.
          -- Damit haben wir die Anzahl von Minuten, die der Rundung vornweg ist.
          -- so z.Bsp 2 wenn Anfangszeit 7:02 und Rundung 15
          -- oder 10 wenn anfangszeit 55 und Rundung 15
        -- Bedeutung trotzdem nicht 100% klar: Rundung ausgehend von Grenze als Nullpunkt?
          -- 09:15 TP-Beginn, Rundung 10 Minuten, Anfang 09:07 wird damit ~ 09:05
        _round_zero_minutes_remainder := extract( minute FROM _round_zero )::numeric % _round_rule;

        _minutes  := extract( minute FROM _date_time - _round_zero_minutes_remainder * '1 minute'::interval );
        _hours    := extract( hour   FROM _date_time - _round_zero_minutes_remainder * '1 minute'::interval );

        _minutes_round_factor := trunc( _minutes / _round_rule ); -- div?

        -- Aufrunden oder Abrunden (Anfangszeit: aufrunden, Endezeit: abrunden)
        IF _round_up THEN
            -- Es gibt einen Rest. Wir haben also einen Fall, in dem aufgerundet werden muss.
            IF ( _minutes / _round_rule ) > _minutes_round_factor THEN -- mod?
                -- Anfangszeit verschiebt sich also um 1 x die Rundungsminuten
                _minutes_round_factor := _minutes_round_factor + 1;
            END IF;

            -- Wir schießen damit über die Stunde hinaus
            IF _minutes_round_factor * _round_rule >= 60 THEN
                -- also 1 zu den Stunden addieren
                _hours := _hours + 1; -- Kann es tatsächlich nur 1 geben?
                _minutes_round_factor := 0; -- was ist mit dem Rest über der Stunde? Sonst ist das automatisch nur eine Rundung bis zur vollen Stunde
            END IF;

        END IF;

        -- Zusammensetzung der gerundeten Zeit
        _date_time_rounded :=
              _date_time::date
            + _hours * '1 hour'::interval
            + ( _minutes_round_factor * _round_rule )::integer * '1 minute'::interval
        ;

        -- Falls wir am Anfang wegen verschobener Anfangs- und Endezeiten eine Abweichung hatten, muss diese jetzt wieder drauf.
        _date_time_rounded :=
              _date_time_rounded
            + _round_zero_minutes_remainder * '1 minute'::interval
        ;

        -- siehe bdep__b_20_iu
        -- Neue Berechnung des ganze Blocks:
          -- new.bd_anf_rund := date_trunc('minute', new.bd_anf) +
              -- Wieviel muss ich zu den Differenzminuten (Plan, Stempel) addieren, damit die Rundung stimmt.
              -- Stichwort: Restklassenring
          -- ((((tplananfmin - date_part('minutes', new.bd_anf))::INTEGER % tplrund) + tplrund) % tplrund) * '1 minute'::INTERVAL;
    END IF;

    -- Rest von Zeit nach Rundung
    _date_time_remainder := _date_time - _date_time_rounded;


    RETURN;
  END $$ LANGUAGE plpgsql IMMUTABLE;

CREATE OR REPLACE FUNCTION TPersonal.bdep__urlaubstage__by_minr_zeitraum__get(
      _minr            integer,
      _datum_von       date,
      _datum_bis       date,
      _abwesenheits_id int,
      _nur_unverbuchte boolean = true
  ) RETURNS numeric AS $$
  DECLARE
      _urlaubstage numeric;
      -- Urlaubstag und halber Urlaubtstag
      _halber_urlaubstag_id CONSTANT int := 100;
  BEGIN

      IF
           _minr IS NULL
        OR _datum_von IS NULL
        OR _datum_bis IS NULL
      THEN
          -- Eingabewerte für Mitarbeiter bzw. Datum sind null.
          RAISE EXCEPTION '%', lang_text( 32638 );
      END IF;

      IF _datum_von > _datum_bis THEN
          -- Bis-Datum ist kleiner als das Von-Datum. Bitte prüfen.
          RAISE EXCEPTION '%', lang_text( 32639 );
      END IF;

      IF NOT EXISTS( SELECT true FROM llv WHERE ll_minr = _minr ) THEN
          -- llv not existing
          RAISE EXCEPTION '%', lang_text( 32638 );
      END IF;

      _urlaubstage :=
             count(1)
        FROM mitpln
        WHERE mpl_minr = _minr
          AND (
                   ( _nur_unverbuchte AND NOT mpl_buch )
                OR NOT _nur_unverbuchte
              )
          AND mpl_date BETWEEN _datum_von AND _datum_bis
          AND mpl_feiertag IS NULL
          AND EXISTS(
                SELECT true
                  FROM bdepab
                 WHERE mpl_date BETWEEN bdab_anf AND bdab_end
                   AND bdab_minr = mpl_minr
                   AND bdab_aus_id = _abwesenheits_id
              )
              ;
      IF _abwesenheits_id = _halber_urlaubstag_id THEN
        _urlaubstage := _urlaubstage / 2;
      END IF;

      RETURN _urlaubstage;
  END $$ LANGUAGE plpgsql;

-- #14772 Urlaubsbuchungen im Zeitraum
CREATE OR REPLACE FUNCTION TPersonal.bdep__urlaub__by_monat__sum__get(
      _minr            integer,
      _datum_von       date,
      _datum_bis       date,
      _nur_unverbuchte boolean = true
  ) RETURNS numeric AS $$
  DECLARE
      _sum             numeric := 0;
      _urlaub_nur_tage boolean;

      -- Urlaubstag und halber Urlaubtstag
      _urlaubstag_id        CONSTANT int := 1;
      _halber_urlaubstag_id CONSTANT int := 100;
  BEGIN

      IF
            _minr IS NULL
         OR _datum_von IS NULL
         OR _datum_bis IS NULL
      THEN
          -- Eingabewerte für Mitarbeiter bzw. Datum sind null.
          RAISE EXCEPTION '%', lang_text( 32638 );
      END IF;

      IF _datum_von > _datum_bis THEN
          -- Bis-Datum ist kleiner als das Von-Datum. Bitte prüfen.
          RAISE EXCEPTION '%', lang_text( 32639 );
      END IF;

      _urlaub_nur_tage := ll_urlaub_days
                         FROM llv
                         WHERE ll_minr = _minr;

      IF _urlaub_nur_tage THEN

          -- Urlaub ganzer Tag
          _sum := TPersonal.bdep__urlaubstage__by_minr_zeitraum__get( _minr, _datum_von, _datum_bis, _urlaubstag_id, _nur_unverbuchte );

          -- Urlaub halbe Tage
          _sum := _sum + TPersonal.bdep__urlaubstage__by_minr_zeitraum__get( _minr, _datum_von, _datum_bis, _halber_urlaubstag_id, _nur_unverbuchte );

      ELSE

          -- Urlaub Stundenweise
          _sum :=      sum( bdab_stu )
                 FROM bdepabsum
                 JOIN mitpln    ON  mitpln.mpl_minr = bdepabsum.mpl_minr
                                AND mitpln.mpl_date = bdepabsum.mpl_date
                 WHERE bdepabsum.mpl_minr=_minr
                   AND (
                         ( _nur_unverbuchte AND NOT mpl_buch )
                          OR
                          NOT _nur_unverbuchte
                       )
                   AND mitpln.mpl_date BETWEEN _datum_von AND _datum_bis
                   AND ab_id IN ( _urlaubstag_id, _halber_urlaubstag_id );
      END IF;

      RETURN coalesce( _sum, 0 );

  END $$ LANGUAGE plpgsql;

-- #14771 Differenz in Zeitraum
CREATE OR REPLACE FUNCTION TPersonal.bdep__saldo_vs_soll__by_monat__differenz__get(
  IN  _minr             integer,
  IN  _datum_von        date,
  IN  _datum_bis        date,
  IN  _nur_unverbuchte  boolean = true
  )
  RETURNS               numeric
  AS $$
  DECLARE
      _sum              numeric := 0;
      _mpl_min          numeric;
      _ll_fixauszahl    numeric;
  BEGIN

      -- ACHTUNG: llv__mitpln__uebersalden__get macht fast das Gleiche. Müßte zusammengelegt werden.

      IF   _minr IS NULL
        OR _datum_von IS NULL
        OR _datum_bis IS NULL
      THEN
          -- Eingabewerte für Mitarbeiter bzw. Datum sind null.
          RAISE EXCEPTION '%', lang_text( 32638 );
      END IF;

      IF _datum_von > _datum_bis THEN
          -- Bis-Datum ist kleiner als das Von-Datum. Bitte prüfen.
          RAISE EXCEPTION '%', lang_text( 32639 );
      END IF;


      SELECT coalesce( sum( mpl_saldo + mpl_absaldo - mpl_min ), 0 ), --gsaldo
             sum( mpl_min ) -- sollzeit für Fixauszahlungen
        INTO _sum,
             _mpl_min
        FROM mitpln
       WHERE mpl_minr = _minr
         AND (
              ( _nur_unverbuchte AND NOT mpl_buch )
                OR
                NOT _nur_unverbuchte
              )
         AND mpl_date BETWEEN _datum_von AND _datum_bis
      ;

      -- Fixauszahlung: der Mitarbeiter bekommt immer X Stunden/Monat ausgezahlt, unabhängig des eigentlichen Monats-Soll (Gehaltsempfänger)
      --  das heißt: der Monat hat 190 Sollstunden. Der Mitarbeiter bekommt im Schnitt über alle Monate 180 Stunden ausgezahlt. (Fixauszahlung 180)
      --  der Mitarbeiter hat 199 Stunden gearbeitet.
      --  Der Mitarbeiter hat also 9 Stunden Überstunden eigentlich. Ausgezahlt bekommt er aber fix 180. Somit hat er nochmal 10 weitere Überstunden, die er nicht ausgezahlt bekommt.
      --  bei einem Monat mit mehr Sollstunden als Fixauszahlung wird entsprechend ein negativer Wert ermittelt. Der Mitarbeiter bekommt mehr ausgezahlt, als er gearbeitet hat
      -- https://ci.prodat-sql.de/sources/tests/suite/18/runner/817

      _ll_fixauszahl := ll_fixauszahl FROM llv WHERE ll_minr = _minr;

      IF _ll_fixauszahl IS NOT null THEN
         _sum :=   _sum  -- überstunden, die sich aus den Tagesplänen ergeben
                 + round( _mpl_min - _ll_fixauszahl, 2); -- differenz aus Sollstunden und Fixauszahlung
      END IF;

      --

      RETURN _sum;

  END $$ LANGUAGE plpgsql;


-- #14769 Funktion zur Ermittlung der neuen LLV-Werte
-- #15419 ERgänzung  Jahresurlaub
CREATE OR REPLACE FUNCTION tpersonal.llv__monatsabschluss_werte__get(
    IN  _minr               integer,
    IN  _datum_von          date,
    IN  _datum_bis          date,
    IN  _nur_unverbuchte    boolean = true,
    IN  _auszahl_stunden    numeric = null,
    IN  _auszahl_urlaub     numeric = null,

    OUT ll_stuko_buch       numeric,   -- Stundenkonto verbucht aus llv
    OUT stuko_diff          numeric,   -- Differenz angegebener Zeitraum
    OUT raucherpause_diff   numeric,   -- Abzug an Raucherpausen
    OUT ll_urlaub_buch      numeric,   -- Urlaub verbucht aus llv
    OUT urlaub_diff         numeric,   -- Differenz Urlaub (Abzug) (Summe aus 1 tag & 1/2 tag Urlaub)
    OUT urlaubauszahl_diff  numeric,   -- Urlaubsauszahlungen
    OUT jahresurlaub        numeric,   -- eventuell draufzuaddierenden Jahresurlaubsanspruch im Januar
    OUT stundauszahl_diff   numeric    -- Stundenauszahlungen
    )
    RETURNS record
    AS  $$
    DECLARE _row_count integer;
            _sa_id integer;
            _monats_stuko numeric;
    BEGIN

        IF     _minr IS null
           OR _datum_von IS null
           OR _datum_bis IS null
        THEN
            -- Eingabewerte für Mitarbeiter bzw. Datum sind null.
            RAISE EXCEPTION '%', lang_text( 32638 );
        END IF;

        IF _datum_von > _datum_bis THEN
            -- Bis-Datum ist kleiner als das Von-Datum. Bitte prüfen.
            RAISE EXCEPTION '%', lang_text( 32639 );
      END IF;

        -- #9566 Automatische Stundenauszahlung
        --  darin auch TSystem.Settings__GetBool('MonAbschlStukoN')
        IF    Settings__GetBool('BDE_Ueberstundenauszahlung')
           OR Settings__GetBool('MonAbschlStukoN')
        THEN
           IF ll_fixauszahl IS NOT null FROM llv WHERE ll_minr = _minr THEN
              -- Fixe Stundenauszahlung ist hier nicht implementiert. Siehe Hinweise in bdep__saldo_vs_soll__by_monat__differenz__get und llv__mitpln__uebersalden__get das diese beiden Funktionen zusammengelegt werden müßten
              RAISE EXCEPTION 'fixauszahlung pro monat und automatische überstundenauszahlung ist nicht implementiert';
           END IF;

           PERFORM TPersonal.llv__stundauszahl__beantragung__create(_minr, _datum_bis);
        END IF;

        -- manuelle Eingaben für Urlaubs - und Stundenauszahlungen. Dann die automatisch generierten Datensätze mit den manuellen Werten überschreiben
          IF _auszahl_stunden IS NOT null THEN
             -- wir haben defintiv einen Wert für auszuzahlende Stunden. <> 0
             UPDATE stundauszahl
                SET sa_stunden = _auszahl_stunden
              WHERE sa_minr = _minr AND sa_date BETWEEN _datum_von AND _datum_bis
                AND sa_type = 'hours';
             GET DIAGNOSTICS _row_count = ROW_COUNT;

             CASE _row_count
                  WHEN 0 THEN
                       IF coalesce(_auszahl_stunden, 0) <> 0 THEN
                          -- Stundenauszahlung > 0 sonst brauchen wir es gar nicht anzulegen!
                          INSERT INTO stundauszahl (sa_stunden, sa_minr, sa_date, sa_type) VALUES (_auszahl_stunden, _minr, _datum_bis, 'hours');
                       END IF;
                  WHEN 1 THEN
                       PERFORM null; -- everything ok
                  ELSE
                       RAISE EXCEPTION 'stundauszahl <> 0,1';
             END CASE;
          END IF;

          IF _auszahl_urlaub IS NOT null THEN -- kein NullIf, da im Update Fall 0 zur Löschung führt!
             -- wir haben defintiv einen Wert für auszuzahlenden Urlaub. <> 0
             UPDATE stundauszahl
                SET sa_urlaub = _auszahl_urlaub
              WHERE sa_minr = _minr AND sa_date BETWEEN _datum_von AND _datum_bis
                AND sa_type = 'holidays';
             GET DIAGNOSTICS _row_count = ROW_COUNT;

             CASE _row_count
                  WHEN 0 THEN
                       -- Urlaub > 0. 0 brauchen wir nicht anzulegen!
                       IF coalesce(_auszahl_urlaub, 0) <> 0 THEN
                          INSERT INTO stundauszahl (sa_urlaub, sa_minr, sa_date, sa_type) VALUES (_auszahl_urlaub, _minr, _datum_bis, 'holidays');
                       END IF;
                  WHEN 1 THEN
                       PERFORM null; -- everything ok
                  ELSE
                       RAISE EXCEPTION 'stundauszahl <> 0,1';
             END CASE;
          END IF;
        --

        -- Bestandsdaten aus llv holen
        SELECT llv.ll_stuko_buch, coalesce( ll_urlaub, 0 )
          INTO ll_stuko_buch,     ll_urlaub_buch
          FROM llv
         WHERE ll_minr = _minr;

        -- differenzen aus angegebenen Zeitraum holen
        urlaub_diff         := coalesce( TPersonal.bdep__urlaub__by_monat__sum__get( _minr, _datum_von, _datum_bis, _nur_unverbuchte ), 0 );
        stuko_diff          := coalesce( TPersonal.bdep__saldo_vs_soll__by_monat__differenz__get( _minr, _datum_von, _datum_bis, _nur_unverbuchte ), 0 );
        raucherpause_diff   := coalesce( TPersonal.llv__bdpab__rpause__round( _minr, _datum_von, _datum_bis, _nur_unverbuchte ), 0 ) ;



        -- stundauszahl : Überstunden und Urlaubsauszahlungen
           -- Stundenkonto immer 0 -> positive Überstunden abschneiden
           IF Settings__GetBool('MonAbschlStukoN') THEN
              _sa_id := sa_id FROM stundauszahl
                             WHERE sa_minr = _minr
                               AND date_to_yearmonth(sa_date) = date_to_yearmonth(_datum_bis)
                               -- ausgenommen Urlaubsfall ~ gleiche Tabelle (Urlaubsauszahlung)
                               AND sa_type = 'hours'
              ;


              -- haben wir ein positives Stundenkonto im Monat, das muss ausgezahlt werden gem. Setting
              -- Stundenkonto liegt unter 0 -> somit auch nichts auszahlen (auch keine Wochenenden/Feiertage welche sonst automatisch kämen)
              _monats_stuko := ll_stuko_buch + stuko_diff - raucherpause_diff;
              _monats_stuko := coalesce( _auszahl_stunden, greatest(0, _monats_stuko), 0 );
              IF _sa_id IS null AND _monats_stuko > 0 THEN
                 INSERT INTO stundauszahl
                             (sa_minr, sa_date, sa_stunden, sa_type)
                      VALUES (_minr, _datum_bis, _monats_stuko, 'hours')
                 ;
              ELSE
              UPDATE stundauszahl
                 SET sa_stunden = _monats_stuko
               WHERE sa_id = _sa_id
                 ;
              END IF;
           END IF;


           -- Hinweis: durch sum() wird auch bei keinem Treffer eine 0 zurückgegeben
           SELECT coalesce( sum( sa_stunden ), 0 ),
                  coalesce( sum( sa_urlaub ), 0)
             INTO stundauszahl_diff,
                  urlaubauszahl_diff
             FROM stundauszahl
            WHERE sa_minr = _minr
              AND sa_date BETWEEN _datum_von AND _datum_bis
              AND (
                      ( _nur_unverbuchte AND NOT sa_buch )
                   OR NOT _nur_unverbuchte
                  )
        ;
        --

        -- Mit Jahresabschluss Dezember den neuen Jahresurlaub
          IF      tpersonal.llv_stuko_history__checkYearDone(_minr, false, _datum_von)
              AND date_part('month', _datum_von) = 12
          THEN --
              jahresurlaub    := coalesce(uan_urlaub_ges, ll_urlaub_ges)
                                   FROM llv
                                   LEFT JOIN llv_url_anspr ON uan_ll_minr = ll_minr
                                                              -- ich schließe das Jahr 2020 ab und bekomme den neuen Jahresurlaub von 2021 aus der Urlaubsstaffel (oder mittels coalesce den standard)
                                                          AND uan_year = date_part('year', _datum_von) + 1
                                  WHERE ll_minr = _minr;
          ELSE
              -- null: kein Jahreswechsel. Übertrag in neues Jahr kann 0
              jahresurlaub    := null;
        END IF;
        --

    END $$ LANGUAGE plpgsql;


-- #14790 Funktion  zum Verbuchen der Monatswerte
CREATE OR REPLACE FUNCTION TPersonal.llv__monatsabschluss__verbuchen(
    IN  _minr                   integer,
    IN  _datum                  date,
    IN  _auszahl_stunden        numeric = null,
    IN  _auszahl_urlaub         numeric = null,
    IN  _ll_stuko_buch_manuell  numeric = null,
    IN  _ll_urlaub_buch_manuell numeric = null
    )
    RETURNS void
    AS $$
    DECLARE
        _yearmonth_datum integer := date_to_yearmonth_dec( _datum );
        _monatsabschluss_werte__get record;
        _r record;
    BEGIN

      IF _minr is null OR _datum is null THEN
        -- Eingabewerte für Mitarbeiter bzw. Datum sind null.
        RAISE EXCEPTION '%', lang_text(32638);
      END IF;


      SELECT last_buch_month__is_previous, yearmonth_dec_to_date(last_buch_month, true) INTO _r
        FROM tpersonal.llv_stuko_history__buch_month__last(_minr, _datum);

      IF NOT _r.last_buch_month__is_previous THEN
        RAISE EXCEPTION 'last_buch_month__is_NOT_previous xtt21749 "%, %"', _minr, _r.yearmonth_dec_to_date;
      END IF;

      -- Step 1: llv aktualisieren. Dazu die Differenzdaten holen und in llv eintragen
      SELECT *
        INTO _monatsabschluss_werte__get
        FROM tpersonal.llv__monatsabschluss_werte__get(_minr,
                                                       date__first_day_of_month__get( _datum ),
                                                       date__last_day_of_month__get( _datum ),
                                                       true,
                                                       _auszahl_stunden,
                                                       _auszahl_urlaub
                          )
      ;

      UPDATE llv SET
                             -- manuelles Überschreiben des Stundenkonto
             ll_stuko_buch = CASE WHEN _ll_stuko_buch_manuell IS null THEN
                                         llv.ll_stuko_buch
                                       + _monatsabschluss_werte__get.stuko_diff
                                       - _monatsabschluss_werte__get.raucherpause_diff
                                       - _monatsabschluss_werte__get.stundauszahl_diff
                                 ELSE
                                       _ll_stuko_buch_manuell
                             END,

                             -- wenn durch jahreswechsel ein neuer Jahresurlaub, dann setzen sonst beibehalten
             ll_urlaub_ges = coalesce(_monatsabschluss_werte__get.jahresurlaub, ll_urlaub_ges),

                             -- Resturlaub nach Verbuchen. Bei Jahreswechsel den neuen Jahresurlaub korrekt berücksichtigen
             ll_urlaub     = CASE WHEN _ll_urlaub_buch_manuell IS null THEN
                                         ll_urlaub
                                       - _monatsabschluss_werte__get.urlaub_diff
                                       - _monatsabschluss_werte__get.urlaubauszahl_diff
                                       + coalesce(_monatsabschluss_werte__get.jahresurlaub, 0)
                                  ELSE
                                         _ll_urlaub_buch_manuell
                                       + coalesce(_monatsabschluss_werte__get.jahresurlaub, 0)
                             END
       WHERE ll_minr = _minr;
      --

      -- Step 2: Eintragen der Abschlußbochung in llv_stuko_history aus llv daten
      INSERT INTO llv_stuko_history (
                  llsh_ll_minr,
                  llsh_monthyear,
                  llsh_urlaub,
                  llsh_stuko,
                  --Erweiterung #21000
                  llsh_stuko_diff,
                  llsh_rp,
                  llsh_sa_stunden,
                  llsh_sa_urlaub
                  )
           SELECT _minr,
                  _yearmonth_datum,
                  -- Der Jahresurlaub wird oben bereits übernommen, daher hier nochmal abziehen für den Verlauf
                  ll_urlaub - coalesce(_monatsabschluss_werte__get.jahresurlaub, 0),
                  ll_stuko_buch,
                  --Erweiterung #21000
                  _monatsabschluss_werte__get.stuko_diff,
                  _monatsabschluss_werte__get.raucherpause_diff,
                  _monatsabschluss_werte__get.stundauszahl_diff,
                  _monatsabschluss_werte__get.urlaubauszahl_diff
             FROM llv
            WHERE ll_minr = _minr
      ;

      -- Jetzt die Sätze als gebucht markieren!
      UPDATE bdep SET
             bd_buch = true
        FROM llv
       WHERE ll_minr = bd_minr
         AND NOT bd_buch
          --aufgrund von Einführung des individuellen Werktags #13388; NEG, JM
         AND date_to_yearmonth_dec( bd_individwt_mpl_date ) <= _yearmonth_datum
         AND bd_minr = _minr;

      UPDATE bdepab SET
             bdab_buch = true
        FROM llv
       WHERE ll_minr = bdab_minr
         AND NOT bdab_buch
         AND date_to_yearmonth_dec( bdab_anf ) <= _yearmonth_datum
         AND bdab_minr = _minr;

      UPDATE stundauszahl SET
             sa_buch = true
        FROM llv
       WHERE ll_minr = sa_minr
         AND NOT sa_buch
         AND date_to_yearmonth_dec( sa_date ) <= _yearmonth_datum
         AND sa_minr = _minr;

      UPDATE mitpln SET
             mpl_buch = true
        FROM llv
       WHERE ll_minr = mpl_minr
         AND NOT mpl_buch
         AND date_to_yearmonth_dec( mpl_date ) <= _yearmonth_datum
         AND mpl_minr = _minr;
        --
    END $$ LANGUAGE plpgsql;
--


CREATE OR REPLACE FUNCTION tpersonal.llv__mitpln__missing_prvday__get(
    IN  _date           date = current_date,
    OUT ll_minr         integer,
    OUT ll_stand_ab_id  integer
    )
    RETURNS SETOF record
    AS $$
        SELECT ll_minr, ll_stand_ab_id
          FROM llv
         WHERE ll_einstd <= _date - 1
           AND coalesce(ll_endd, _date) >= _date - 1
           AND ll_autostemp
           AND ll_prszzeit
           -- es existiert kein mitpln-eintrag
           AND NOT exists(SELECT true FROM mitpln WHERE mpl_date = _date -1 AND mpl_minr = ll_minr)
           -- prüfen, das 1 tag zurück nicht vor dem Monatsabschluss liegt
           AND check_valid_bde_date(ll_minr, _date - 1)
           -- nur Arbeitstage
           AND extract(DOW FROM _date - 1) BETWEEN 1 AND 5;
    $$ LANGUAGE sql;


-- Kontext: Urlaubsverfall zurückgeben >> Resturlaub muss bis zu einem Stichtag verbraucht sein
-- Beispiel Datum bei 3 Monate Warn-Datum und aktuellem Jahr 2024
  -- 01.12.24 heute und 3 Monate Warndatum = 31.03.25 - 3 Monate = 31.12.24 > 01.12.24 >> keine Ausgabe
  -- 01.01.25 heute und 3 Monate Warndatum = 31.12.24 > 01.01.25 nein  >> Berechnung findet statt
CREATE OR REPLACE FUNCTION tpersonal.vacation__expiration__days__get(
    _minr integer,             -- Mitarbeiternummer
    _date date = current_date  -- Datum des aktuellen Tages
    ) RETURNS numeric AS $$
    DECLARE
      date_exp              DATE;
      date_exp_start        DATE;
      date_warn             DATE;
      llv__urlaub__act      NUMERIC;
      days_exp              NUMERIC;
      is_docker_or_postgres BOOLEAN := current_user IN ('docker', 'postgres');
    BEGIN

      IF ( TSystem.Settings__Get('BDE_URLAUBSVERFALL') = 1) THEN -- Stichtag: der 31.03. ~ default
        -- Stichtag im gleichen Jahr wie Eingabedatum
        date_exp := MAKE_DATE(EXTRACT(YEAR FROM _date)::INTEGER, 3, 31);
        -- Wenn das Eingabedatum höher als Stichtag im gleichen Jahr ist, gilt der Stichtag des Folgejahres
        IF _date > date_exp THEN
          date_exp := MAKE_DATE(EXTRACT(YEAR FROM _date)::INTEGER + 1, 3, 31); -- Stichtag: der 31.03. des Folgejahres ~ default
        END IF;

      ELSE
        IF is_docker_or_postgres THEN
          RAISE NOTICE 'Option nicht gesetzt oder nicht implementiert.';
        END IF;
        RETURN 0; --später Option 2 implementieren date_exp := MAKE_DATE(EXTRACT(YEAR FROM _date)::INTEGER, 12, 31); -- der letzte Tag des aktuelle Jahres

      END IF;

      date_exp_start := DATE_TRUNC('year', date_exp)::DATE;  -- 01.01. Stichtagsjahr

      -- vom Stichtag ziehen wir die Monate ab, die im Setting angeben sind
      date_warn := date_exp - (CAST(TSystem.Settings__Get('BDE_URLAUBSVERFALL_WARNUNG') AS INTEGER) || ' months')::INTERVAL;

      -- Das Warn-Datum ist erreicht: Ausgabe evtl. Verfalls-Tage
      IF ( date_warn < _date  ) THEN

        ---Raise Notice '1 Eingabedatum (%) >> date_exp (%); date_warn (%) < _date (%)', _date, date_exp, date_warn, _date ;

        --Rest-Urlaubstage aus dem Jahr vor dem Stichtag ABER OHNE Zuschlag neuer Jahresurlaub, wie es tpersonal.llv__urlaub__act macht
        llv__urlaub__act := tpersonal.llv__urlaub__rest__by__pastdate__get(_minr, date_exp_start, true);

        -- Wenn das Warn-Datum im gleichen Jahr wie das Eingabedatum ist, dann befinden wir uns noch nicht im Stichtags-Jahr ~ relevant für Berechnungsvorschrift
        IF ( DATE_PART('year', date_warn) = DATE_PART('year', _date) ) THEN
          -- Wir sind im Vorjahr des Stichtages bspw. bei 4 Monate Warnung : 01.12.24 und heute ist 5.12.24

          ---Raise Notice '0.1 Wir sind im Vorjahr des Stichtages >> llv__urlaub__act (%)', llv__urlaub__act ;

          IF ( llv__urlaub__act > 0 ) THEN -- Es gibt Resturlaub aus dem Jahr vor dem Stichtag

            -- Jahr des Datums (Urlaubstage des aktuellen Jahres bis Datum + geplante Tage ab Datum) + Geplante Tage nächstes Jahr
            -- Die Verfalls-Tage berechnen:
            days_exp := (   llv__urlaub__act                                                                 -- Urlaubstage des aktuellen Jahres bis Datum
                          - tpersonal.llv__urlaub__geplant__by__currentyear__get(_minr, true, false, _date)  -- geplante Tage ab Datum
                        ) -- _date-Jahr
                        - ( tpersonal.llv__urlaub__geplant__by__currentyear__get(_minr, true, false, date_exp_start)  -- Tage gesamtes Jahr
                            - tpersonal.llv__urlaub__geplant__by__currentyear__get(_minr, true, false, date_exp)      -- Tage ab 31.03
                          ) -- Stichtags-Jahr
                        ;

            IF ( days_exp > 0) THEN
              RETURN days_exp;
            ELSE
              IF is_docker_or_postgres THEN
                RAISE NOTICE 'Resturlaub genommen.';
              END IF;
              RETURN 0;
            END IF;

          ELSE
            IF is_docker_or_postgres THEN
              RAISE NOTICE 'Kein Resturlaub zu verbrauchen';
            END IF;
            RETURN 0; -- Es gibt keinen Resturlaub
          END IF;

        ELSE -- Wir sind im Stichtags-Jahr (Warnverfall max. 3 Monate)

          ---Raise Notice '1.1 Wir sind im Stichtags-Jahr >> llv__urlaub__act (%)', llv__urlaub__act ;

          IF ( llv__urlaub__act > 0 ) THEN -- Es gibt Resturlaub aus dem Jahr vor dem Stichtag

            -- Die Verfalls-Tage berechnen: Rest-Urlaubstage des Vorjahres - genehmigte Urlaubstage seit 01.01 bis 31.03 Stichtagsjahr
            -- An dieser Stelle wäre es besser man musste die Funktion nicht 2 mal aufrufen, würde die Änderung von llv__urlaub__geplant__by__currentyear__get erfordern
            days_exp := llv__urlaub__act -- Rest-Urlaub im Jahr vor Stichtag
                        - (  tpersonal.llv__urlaub__geplant__by__currentyear__get(_minr, true, false, date_exp_start)  -- Tage gesamtes Jahr
                           - tpersonal.llv__urlaub__geplant__by__currentyear__get(_minr, true, false, date_exp)        -- Tage ab 31.03
                          )
                        ;

            IF ( days_exp > 0) THEN
              RETURN days_exp;
            ELSE
              IF is_docker_or_postgres THEN
                RAISE NOTICE 'Resturlaub genommen.';
              END IF;
              RETURN 0;
            END IF;
          ELSE
            IF is_docker_or_postgres THEN
              RAISE NOTICE 'Kein Vorjahres-Resturlaub';
            END IF;
            RETURN 0; -- Es gibt keinen Resturlaub
          END IF;

        END IF;

      ELSE -- Warn-Datum noch nicht erreicht
        IF is_docker_or_postgres THEN
          RAISE NOTICE 'Warn-Datum noch nicht erreicht';
        END IF;
        ---Raise Notice '2 Eingabedatum (%) >> date_exp (%); date_warn (%) > _date (%)', _date, date_exp, date_warn, _date ;
        RETURN 0;
      END IF;

      IF is_docker_or_postgres THEN
        RAISE NOTICE 'Keine Einstiegsbedingungen erfüllt.';
      END IF;
      RETURN 0;
    END $$ LANGUAGE plpgsql STABLE;
--

----------------------------------- zentrale Auftragszeit-Funktionen Anfang ----------------------------------

-- PROTOTYP siehe #15684
CREATE OR REPLACE FUNCTION TPersonal.bdea__eintrag__execute(
      _minr                     integer,          -- Mitarbeiter-Nr.
      _abk_index                integer,          -- ABK-Index
      _arbeitsgang              integer,          -- ABK-Arbeitsgang
      _ruesten                  boolean = false,  -- Rüsten anstempeln
      _kostenstelle             varchar = null,   -- Kostenstelle
      _arbeitsplatz             varchar = null,   -- Arbeitsplatz an Kostenstelle
      _menge_gefertigt          numeric = null,   -- Menge der gefertigten Teile
      _menge_ausschuss          numeric = null,   -- Menge an Ausschuss
      _ausschussgrund_id        integer = null,   -- Grund für Ausschuss
      _ausfallgrund_id          integer = null,   -- Ausfallgrund (auch Pause/Raucherpause)
      _arbeitsgang_ende         boolean = false,  -- Arbeitsgang wird beendet
      _mehrmaschinenbedienung   boolean = false,  -- Mehrmaschinenbedienung (effektive Auftragszeit = Stückzahl * Stückzeit)
      _stueckzeit               numeric = null,   -- Zeit pro Stück in h
      _effektive_zeit           numeric = null,   -- effektive Auftragszeit in h
      _restzeit                 numeric = null,   -- Info bzgl. aktueller Restarbeiszeit
      _stempel_info             text = null,      -- Hinweis an Stempelung
      _stempel_zeitpunkt_start  timestamp = currenttime(),  -- Stempelzeit (Start der Auftragszeit)
      _stempel_zeitpunkt_ende   timestamp = null -- Stempelzeit (Ende der Auftragszeit)
  ) RETURNS void AS $$
  BEGIN
      -- PROTOTYP siehe #15684
      -- zentrale Funktion für das Eintragen einer Auftragszeit
        -- Mitarbeiter-Nr. darf leer sein. Aufwand ohne Mitarbeiterzuordnung. Muss aber explizit mit null angg. werden.
        -- Ohne Angabe von Kostenstelle wird die Vorgabe ermittelt.
        -- Keine Ermittlung der Vorgabe für Arbeitsplatz.


      -- Fehler: Ohne Angabe von ABK, AG oder Stempelzeit
      IF ( _abk_index IS NULL OR _arbeitsgang IS NULL OR _stempel_zeitpunkt_start IS NULL ) THEN
          RAISE EXCEPTION '%', lang_text( 16863 );
      END IF;


      -- Kostenstelle automatisch ermitteln
      IF _kostenstelle IS NULL THEN

          _kostenstelle :=
                a2_ks
              FROM ab2
              WHERE a2_ab_ix = _abk_index
                AND a2_n = _arbeitsgang
          ;

      END IF;


      -- Auftragszeit eintragen
      INSERT INTO bdea (
          ba_minr,            ba_ix,                    ba_op,
          ba_ruest,           ba_ks,                    ba_ksap,
          ba_stk,             ba_auss,
          ba_asg_id,          ba_aus_id,
          ba_ende,            ba_efftime_calc_by_stk,
          ba_stk_time,        ba_efftime,               ba_resttime,
          ba_txt,
          ba_anf,
          ba_end
        )
      VALUES (
          _minr,              _abk_index,               _arbeitsgang,
          _ruesten,           _kostenstelle,            _arbeitsplatz,
          _menge_gefertigt,   _menge_ausschuss,
          _ausschussgrund_id, _ausfallgrund_id,
          _arbeitsgang_ende,  _mehrmaschinenbedienung,
          _stueckzeit,        _effektive_zeit,          _restzeit,
          _stempel_info,
          _stempel_zeitpunkt_start,
          _stempel_zeitpunkt_ende
        )
      ;


      RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;
--

-- PROTOTYP siehe #15684
CREATE OR REPLACE FUNCTION TPersonal.bdea__stempeln__start__execute(
      _minr                     integer,          -- Mitarbeiter-Nr.
      _abk_index                integer,          -- ABK-Index
      _arbeitsgang              integer,          -- ABK-Arbeitsgang
      _ruesten                  boolean = false,  -- Rüsten anstempeln
      _kostenstelle             varchar = null,   -- Kostenstelle
      _arbeitsplatz             varchar = null,   -- Arbeitsplatz an Kostenstelle
      _stempel_info             text = null,      -- Hinweis an Stempelung
      _stempel_zeitpunkt_start  timestamp = currenttime() -- Stempelzeit (Start der Auftragszeit)
  ) RETURNS void AS $$
  BEGIN
      -- PROTOTYP siehe #15684
      -- zentrale Funktion für das Anstempeln der Auftragszeit
        -- siehe Hinweise TPersonal.bdea__eintrag__execute


      -- Auftragszeit-Start eintragen
      PERFORM
          TPersonal.bdea__eintrag__execute(
              _minr,
              _abk_index,
              _arbeitsgang,
              _ruesten,
              _kostenstelle,
              _arbeitsplatz,
              _stempel_info => _stempel_info,
              _stempel_zeitpunkt_start => _stempel_zeitpunkt_start
          )
      ;


      RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;
--

-- PROTOTYP siehe #15684
CREATE OR REPLACE FUNCTION TPersonal.bdea__stempeln__nachtrag__execute(
      _minr                     integer,          -- Mitarbeiter-Nr.
      _abk_index                integer,          -- ABK-Index
      _arbeitsgang              integer,          -- ABK-Arbeitsgang
      _ruesten                  boolean = false,  -- Rüsten anstempeln
      _kostenstelle             varchar = null,   -- Kostenstelle
      _arbeitsplatz             varchar = null,   -- Arbeitsplatz an Kostenstelle
      _menge_gefertigt          numeric = null,   -- Menge der gefertigten Teile
      _menge_ausschuss          numeric = null,   -- Menge an Ausschuss
      _ausschussgrund_id        integer = null,   -- Grund für Ausschuss
      _arbeitsgang_ende         boolean = false,  -- Arbeitsgang wird beendet
      _mehrmaschinenbedienung   boolean = false,  -- Mehrmaschinenbedienung (effektive Auftragszeit = Stückzahl * Stückzeit)
      _stueckzeit               numeric = null,   -- Zeit pro Stück in h
      _effektive_zeit           numeric = null,   -- effektive Auftragszeit in h
      _restzeit                 numeric = null,   -- Info bzgl. aktueller Restarbeiszeit
      _stempel_info             text = null,      -- Hinweis an Stempelung
      _stempel_zeitpunkt_anfang timestamp = currenttime(),  -- Stempelzeit (Start der Auftragszeit)
      _stempel_zeitpunkt_ende   timestamp = null -- Stempelzeit (Ende der Auftragszeit)
  ) RETURNS void AS $$
  BEGIN
      -- PROTOTYP siehe #15684
      -- zentrale Funktion für das Nachtragen einer Auftragszeit
        -- siehe Hinweise TPersonal.bdea__eintrag__execute
        -- Ausfallgrund kann nicht angg. werden.


      -- Auftragszeit nachtragen
      PERFORM
          TPersonal.bdea__eintrag__execute(
              _minr,
              _abk_index,
              _arbeitsgang,
              _ruesten,
              _kostenstelle,
              _arbeitsplatz,
              _menge_gefertigt,
              _menge_ausschuss,
              _ausschussgrund_id,
              null,
              _arbeitsgang_ende,
              _mehrmaschinenbedienung,
              _stueckzeit,
              _effektive_zeit,
              _restzeit,
              _stempel_info,
              _stempel_zeitpunkt_anfang,
              _stempel_zeitpunkt_ende
          )
      ;


      RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;
--

-- PROTOTYP siehe #15684
CREATE OR REPLACE FUNCTION TPersonal.bdea__stempeln__ende__execute(
      _bdea_id                integer,          -- Auftragszeit (welche beendet wird)
      _menge_gefertigt        numeric = null,   -- Menge der gefertigten Teile
      _menge_ausschuss        numeric = null,   -- Menge an Ausschuss
      _ausschussgrund_id      integer = null,   -- Grund für Ausschuss
      _ausfallgrund_id        integer = null,   -- Ausfallgrund (auch Pause/Raucherpause)
      _arbeitsgang_ende       boolean = false,  -- Arbeitsgang wird beendet
      _mehrmaschinenbedienung boolean = false,  -- Mehrmaschinenbedienung (effektive Auftragszeit = Stückzahl * Stückzeit)
      _stueckzeit             numeric = null,   -- Zeit pro Stück in h
      _restzeit               numeric = null,   -- Info bzgl. aktueller Restarbeiszeit in h
      _stempel_info           text = null,      -- Hinweis an Stempelung
      _stempel_zeitpunkt_ende timestamp = currenttime() -- Stempelzeit (Ende der Auftragszeit)
  ) RETURNS void AS $$
  BEGIN
      -- PROTOTYP siehe #15684
      -- zentrale Funktion für das Abstempeln und Unterbrechen (auch Pause/Raucherpause) einer spezifischen Auftragszeit
        -- Bei Angabe von _stempel_info wird vorhandener Hinweis überschrieben, bei null bleibt er erhalten.
        -- Rüsten kann nicht umgeschrieben werden, da dies per Anstempeln der Auftragszeit festgelegt wird.
        -- Effektive Auftragszeit kann nicht übergeben werden, da diese berechnet wird.


      -- Fehler: Ohne Angabe von Auftrags- oder Stempelzeit
      IF ( _bdea_id IS NULL OR _stempel_zeitpunkt_ende IS NULL ) THEN
          RAISE EXCEPTION '%', lang_text( 16864 );
      END IF;


      -- Auftragszeit beenden
      UPDATE bdea SET
        ba_stk                  = _menge_gefertigt,
        ba_auss                 = _menge_ausschuss,
        ba_asg_id               = _ausschussgrund_id,
        ba_ende                 = _arbeitsgang_ende,
        ba_efftime_calc_by_stk  = _mehrmaschinenbedienung,
        ba_stk_time             = _stueckzeit,
        ba_resttime             = _restzeit,
        ba_aus_id               = _ausfallgrund_id,
        ba_txt                  = coalesce( _stempel_info, ba_txt ),
        ba_end                  = _stempel_zeitpunkt_ende
      WHERE ba_id = _bdea_id
      ;


      RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;
--

--
CREATE OR REPLACE FUNCTION tpersonal.bde__bdea__offene(
      _minr   integer -- Mitarbeiternummer
  ) RETURNS SETOF bdea AS $$
    -- Ermittelt alle offenen Auftragszeiten des Mitarbeiters.


    SELECT bdea
      FROM bdea
    WHERE ba_minr = _minr
      AND NOT ba_ende
      AND ba_end     IS NULL
      AND ba_efftime IS NULL
    ORDER BY ba_anf, ba_ix, ba_op, ba_id

  $$ LANGUAGE sql STABLE;
--

--
CREATE OR REPLACE FUNCTION tpersonal.bde__bdea__stempeln__ende(
      _ba_id                              integer,        -- ID der Auftragszeit (bdea)
      _stempel_zeitpunkt                  timestamp,      -- Zeitpunkt der Ausstempelung
      _ba_aus_id                          integer = null, -- Ausfallgrund der Ausstempelung (Pause, Raucherpause, Schaden)

      _by__ksv__ks_bdea_pause_interrupts  boolean = true, -- Option: Berücksichtigung der KS-Konfiguration bzgl.
                                                          -- automatischer Unterbrechnung
      _bdea__offen__only                  boolean = true  -- Option: ausschließlich offene Auftragszeiten
                                                          -- oder Status irrelevant
  ) RETURNS void AS $$
  BEGIN
    -- Stempelt genau eine Auftragszeit aus.
      -- Zeitpunkt und Ausfallgrund der Ausstempelung.
      -- Option: Berücksichtigung der Kostenstellen-Konfiguration bzgl. automatischer Unterbrechnung
      -- Option: ausschließlich offene Auftragszeiten oder Status irrelevant


    -- Fehler: Ohne Angabe von ID oder Ende der Auftragszeit
    IF ( _ba_id IS NULL OR _stempel_zeitpunkt IS NULL ) THEN
        RAISE EXCEPTION '%', lang_text(16838);
    END IF;

    -- Auftragszeit beenden
    UPDATE bdea SET
      ba_end    = _stempel_zeitpunkt,
      ba_aus_id = _ba_aus_id
    WHERE ba_id = _ba_id

      -- Option: ausschließlich offene Auftragszeiten oder Status irrelevant
      AND (
            -- Status irrelevant
            _bdea__offen__only IS FALSE -- NULL wird wie true behandelt (Default)

            -- nur offene Auftragszeiten
            OR  (
                    NOT ba_ende
                AND ba_end     IS NULL
                AND ba_efftime IS NULL
            )
      )

      -- Option: Berücksichtigung der Kostenstellen-Konfiguration bzgl. automatischer Unterbrechnung
      AND (
            -- alle ohne Berücksichtung der KS-Konfig
            _by__ksv__ks_bdea_pause_interrupts IS FALSE -- NULL wird wie true behandelt (Default)

            -- entspr. KS-Konfig, ob autom. Unterbrechung stattfinden soll.
            OR  EXISTS (
                    SELECT true
                    FROM ksv
                    WHERE ks_abt = ba_ks
                      AND ks_bdea_pause_interrupts
                )
      )
    ;


    RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;
--

--
CREATE OR REPLACE FUNCTION tpersonal.bde__bdea__stempeln__ende__offene(
      _minr                               integer,        -- Mitarbeiternummer
      _stempel_zeitpunkt                  timestamp,      -- Zeitpunkt der Ausstempelung
      _ba_aus_id                          integer = null, -- Ausfallgrund der Ausstempelung (Pause, Raucherpause, Schaden)

      _by__ksv__ks_bdea_pause_interrupts  boolean = true  -- Option: Berücksichtigung der KS-Konfiguration
                                                          -- bzgl. automatischer Unterbrechnung
  ) RETURNS void AS $$
  BEGIN
    -- Stempelt alle offenen Auftragszeien eines Mitarbeiter aus.
      -- Zeitpunkt und Ausfallgrund der Ausstempelung.
      -- Option: Berücksichtigung der Kostenstellen-Konfiguration bzgl. automatischer Unterbrechnung


    -- Fehler: Ohne Angabe von Mitarbeiter-Nr. oder Ende der Auftragszeit
    IF ( _minr IS NULL OR _stempel_zeitpunkt IS NULL ) THEN
        RAISE EXCEPTION '%', lang_text(16839);
    END IF;

    -- Alle offenen Auftragszeiten des MA ermitteln und beenden
    PERFORM
      tpersonal.bde__bdea__stempeln__ende(
          bde__bdea__offene.ba_id,
          _stempel_zeitpunkt,
          _ba_aus_id,
          _by__ksv__ks_bdea_pause_interrupts
      )
    FROM  -- alle offenen Auftragszeien des MA
          tpersonal.bde__bdea__offene( _minr )
    ;


    RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;
--

----------------------------------- zentrale Auftragszeit-Funktionen Ende ------------------------------------

-------- #19891 Begin

-- Neue Basis-Funktion zum Prüfen, ob meine Abteilung #19891
-- Ergänzung zu tpersonal.llv__user_rights_personal__mine_or_abteilunsgleiter
CREATE OR REPLACE FUNCTION tpersonal.llv__user_abteilung__is_mine(
  _usename varchar,                 -- Benutzer, welcher Zugreifen möchte ~ viewer
  _llv llv                          -- Benutzer - Daten, auf welchen zugegriffen werden soll ~ user data
) RETURNS boolean AS $$

DECLARE
  _is_same_abteilung boolean = false;
  _abteilung         varchar;
  _usenameexists     boolean;

BEGIN
  IF ( _usename IS NULL ) THEN
    RETURN false;
  END IF;

  IF ( _llv IS NULL) THEN
    RETURN false;
  END IF;

  -- Abteilung des Viewers holen
  -- KEIN COALESCE(ll_abteilung), dies würde dazu führen dass alle ohne Abteilung (null) als eigenständige Abteilung behandelt würden
  SELECT
    true,
    llv_viewer.ll_abteilung
  FROM
    llv AS llv_viewer
  WHERE
    llv_viewer.ll_db_usename = _usename
  INTO
    _usenameexists,
    _abteilung
  ;

  IF (_usenameexists IS NULL) THEN
    RETURN false;
  END IF;

  IF ( _llv.ll_abteilung IS NOT NULL )  THEN
    -- Abteilung des Viewers stimmt mit Abteilung des aktuellen Nutzers überein
    _is_same_abteilung := ( _abteilung IS NOT NULL ) AND ( _abteilung = _llv.ll_abteilung );

    -- Wenn Abteilung nicht zusammenpasst, dann prüfen wir ob 'Abteilungsleiter für mehrer Abteilungen' definiert sind
    IF NOT _is_same_abteilung THEN
      SELECT
        EXISTS
        (
          SELECT
            llv_abteilungen.lla_abteilung
          FROM
            llv_abteilungen
          WHERE
            ( llv_abteilungen.lla_abteilung = _llv.ll_abteilung )
            AND
            ( llv_abteilungen.lla_db_usename = _usename )
        )
      INTO
        _is_same_abteilung
      ;
    END IF;
  END IF;

  RETURN _is_same_abteilung;
END
$$ LANGUAGE plpgsql STABLE;

CREATE OR REPLACE FUNCTION tpersonal.llv__user_abteilung__is_mine(
  _usename     varchar, -- Benutzer welcher Zugreifen möchte
  _for_usename varchar  -- Benutzer ´(seine Daten) auf welchen zugegriffen werden soll
) RETURNS boolean AS $$

DECLARE
  _is_same_abteilung boolean;

BEGIN
  _is_same_abteilung := false;

  if ( ( _usename IS NOT NULL ) AND ( _for_usename IS NOT NULL ) ) THEN
    SELECT
      tpersonal.llv__user_abteilung__is_mine(_usename, llv)
    FROM
      llv
    WHERE
      (llv.ll_db_usename = _for_usename)
    INTO
      _is_same_abteilung
    ;
  END IF;

  RETURN COALESCE(_is_same_abteilung, false);
END $$ LANGUAGE plpgsql STABLE;

-------- #19891 End


-------- #16744 Begin
-- #19352

CREATE OR REPLACE FUNCTION tpersonal.llv__user_rights_personal__mine_or_abteilunsgleiter(
  _usename varchar,                 -- Benutzer welcher Zugreifen möchte
  _llv llv,                         -- Benutzer ´(seine Daten) auf welchen zugegriffen werden soll
  _ignore_personal boolean = false  -- Sys.Personal bei Auswertung ignorieren
) RETURNS boolean AS $$

DECLARE
  --
  _is_self           boolean;
  _is_same_abteilung boolean;
  --
  _abteilung         varchar;
  _abteilungsleiter  boolean;

BEGIN
  IF ( _usename IS NULL ) THEN
    RETURN false;
  END IF;

  IF ( (_ignore_personal = false) and TSystem.roles__user__rights_personal( _usename, '' ) ) THEN
    RETURN true;
  END IF;

  IF ( _llv IS NULL) THEN
    RETURN false;
  END IF;

  _is_self           := ( ( _llv.ll_db_usename IS NOT NULL ) ) AND ( _usename = _llv.ll_db_usename );
  _is_same_abteilung := false;
  _abteilungsleiter  := false;

  IF NOT _is_self THEN
    -- Daten für den 'Editor' holen
    -- KEIN COALESCE(ll_abteilung), dies würde dazu führen dass alle ohne Abteilung (null) als eigenständige Abteilung behandelt würden
    SELECT
      llv_abteilungsleiter.ll_abteilung,
      llv_abteilungsleiter.ll_abteilungsleiter
    FROM
      llv AS llv_abteilungsleiter
    WHERE
      ( llv_abteilungsleiter.ll_db_usename = _usename )
    INTO
      _abteilung,
      _abteilungsleiter
    ;

    IF ( _llv.ll_abteilung IS NOT NULL )  THEN
      _is_same_abteilung :=
      (
        ( _abteilung IS NOT NULL )
        AND
        ( _abteilung = _llv.ll_abteilung )
      );

      IF NOT _is_same_abteilung THEN
        SELECT
          (
            EXISTS
            (
              SELECT
                llv_abteilungen.lla_abteilung
              FROM
                llv_abteilungen
              WHERE
                (
                  ( llv_abteilungen.lla_abteilung = _llv.ll_abteilung )
                  AND
                  ( llv_abteilungen.lla_db_usename = _usename )
                )
            )
          )
        INTO
          _is_same_abteilung
        ;
      END IF;
    END IF;
  END IF;

  RETURN
  (
    ( _is_self )
    OR
    (
      ( (_abteilungsleiter IS NOT NULL) AND ( _abteilungsleiter ) )
      AND
      ( _is_same_abteilung )
    )
  );
END
$$ LANGUAGE plpgsql STABLE;

-------- #16744 End


-- #16900 (#16771) BEGIN
CREATE OR REPLACE FUNCTION tpersonal.bde__visibility_abwbewill(
  _usename varchar, -- Benutzer welcher Zugreifen möchte
  _llv llv          -- Benutzer (seine Daten) auf welchen zugegriffen werden soll
) RETURNS boolean AS $$

DECLARE
  _visible             boolean;
  _abteilung           varchar;
  _abteilungsleiter    boolean;
  _urlaub_bewill_vorl  boolean;
  _urlaub_bewill       boolean;
  _abteilung_in_llvabt boolean;
  _has_some_rights     boolean;
  _usenameexists       boolean;

BEGIN
  IF ( ( _usename IS NULL ) OR ( _llv.ll_db_usename IS NULL ) ) THEN
    RETURN false;
  END IF;

  -- Setting-Name irreführend, da ursprünglich dafür gedacht,
  -- die Sichtbarkeit von "nicht eigenen" Datensätzen auf AL zu beschränken
  -- gemeint ist für BDE und AbwBew:
  -- dass in diesem Fall die Berechtigungen auf Abteilungen eingeschränkt werden
  -- und damit die Sichtbarkeit gewährleistet sein muss
  _visible := ( NOT(TSystem.Settings__GetBool( 'BDE_bdep_abteilungsleiter_rights' ) ) )
              OR
              ( TSystem.roles__user__rights_personal(_usename, '') );

  IF NOT _visible THEN
    -- Abteilungsleiter, welche sich nicht selbst bewilligen dürfen,
    -- werden durch Sys.Personaldaten.Abwesenheiten.ALohneRecht genehmigt
    _visible := _llv.ll_urlaub_bewill_notself AND ( TSystem.roles__user__rights_personal( _usename, 'AbwBewill.ALohneRecht' ) );

    IF NOT _visible THEN
      -- jeder Nutzer darf die ihn betreffenden Einträge sehen
      _visible := ( _llv.ll_db_usename = _usename );
      IF NOT _visible THEN
        SELECT
          true,
          -- KEIN COALESCE(ll_abteilung), dies würde dazu führen dass alle ohne Abteilung (null) als eigenständige Abteilung behandelt würden
          llv_abteilungsleiter.ll_abteilung,
          COALESCE( llv_abteilungsleiter.ll_abteilungsleiter, false ),
          COALESCE( llv_abteilungsleiter.ll_urlaub_bewill_vorl, false ),
          COALESCE( llv_abteilungsleiter.ll_urlaub_bewill, false )
        FROM
          llv AS llv_abteilungsleiter
        WHERE
          ( llv_abteilungsleiter.ll_db_usename = _usename )
        INTO
          _usenameexists,
          _abteilung,
          _abteilungsleiter,
          _urlaub_bewill_vorl,
          _urlaub_bewill
        ;

        IF (_usenameexists IS NULL) THEN
          RETURN false;
        END IF;

        _has_some_rights :=
        (
          ( _urlaub_bewill )
          OR
          (
            ( TSystem.Settings__GetBool( 'BDE_Abwesenheitsbeantragung_Mehrstufig' ) )
            AND
            ( _urlaub_bewill_vorl )
          )
        );

        -- Abteilungsleiter dürfen Ihre eigene Abteilung sehen
        -- MA dürfen Ihre eigene Abteilung sehen, sofern sie AbwBew genehmigen dürfen
        _visible :=
        (
          ( _abteilungsleiter )
          OR
          ( _has_some_rights )
        )
        AND
        ( ( _llv.ll_abteilung IS NOT NULL ) AND ( _abteilung IS NOT NULL ) AND ( _abteilung = _llv.ll_abteilung ) );

        IF NOT _visible THEN
          SELECT
            (
              EXISTS
              (
                SELECT
                  llv_abteilungen.lla_abteilung
                FROM
                  llv_abteilungen
                WHERE
                  (
                    ( _llv.ll_abteilung IS NOT NULL )
                    AND
                    ( llv_abteilungen.lla_abteilung = _llv.ll_abteilung )
                    AND
                    ( llv_abteilungen.lla_db_usename = _usename )
                  )
              )
            )
          INTO
            _abteilung_in_llvabt
          ;

          _visible :=
          (
            -- Abteilungsleiter dürfen andere Abteilungen sehen, sofern diese in llv_abteilungen definiert sind
            ( _abteilung_in_llvabt AND _abteilungsleiter )
            OR
            -- Nicht-Abteilungsleiter dürfen andere Abteilungen sehen,
            -- sofern diese in llv_abteilungen definiert sind UND Sie Urlaub bewilligen dürfen ('AL Light')
            ( _abteilung_in_llvabt AND NOT( _abteilungsleiter ) AND ( _has_some_rights ) )
          );
        END IF;
      END IF;
    END IF;
  END IF;

  RETURN COALESCE(_visible, false);
END $$ LANGUAGE plpgsql STABLE;

CREATE OR REPLACE FUNCTION tpersonal.bde__visibility_abwbewill(
  _usename     varchar, -- Benutzer welcher Zugreifen möchte
  _for_usename varchar  -- Benutzer ´(seine Daten) auf welchen zugegriffen werden soll
) RETURNS boolean AS $$

DECLARE
  _visible boolean;

BEGIN
  _visible := false;

  if ( ( _usename IS NOT NULL ) AND ( _for_usename IS NOT NULL ) ) THEN
    SELECT
      tpersonal.bde__visibility_abwbewill(_usename, llv)
    FROM
      llv
    WHERE
      (llv.ll_db_usename = _for_usename)
    INTO
      _visible
    ;
  END IF;

  RETURN COALESCE(_visible, false);
END $$ LANGUAGE plpgsql STABLE;

CREATE OR REPLACE FUNCTION tpersonal.bde__user_rights_operation(
  _usename                varchar,                         -- Benutzer welcher Zugreifen möchte
  _llv                    llv,                             -- Benutzer ´(seine Daten) auf welchen zugegriffen werden soll
  _operation              TPersonal.bde__abwbew_operation, -- Operation für die Recht ermittelt werden soll
  _is_marked_for_approval boolean,                         -- Bereits vorläufig genehmigt (mehrstufige Bewilligung)
  _is_approved            boolean,                         -- Bereits genehmigt
  _is_denied              boolean                          -- Bereits abgelehnt
) RETURNS boolean AS $$

DECLARE
  -- settings
  _is_al_restricted_rights boolean;
  _is_markfa_needed        boolean;
  -- berechnete
  _is_supergroup           boolean;
  _is_self                 boolean;
  _is_same_abteilung       boolean;
  _is_al_of                boolean;
  _is_marked_fa            boolean;
  _is_finalized            boolean;
  -- 'Editor' Eigenschaften
  _abteilung               varchar;
  _abteilungsleiter        boolean;
  _urlaub_bewill_vorl      boolean;
  _urlaub_bewill           boolean;
  _urlaub_notself          boolean;
  _editorexists            boolean;
  -- temporäre
  _has_right_1             boolean;
  _has_right_2             boolean;

BEGIN
  if (
       ( _usename IS NULL ) OR (_llv.ll_db_usename IS NULL)
       OR
       ( _operation IS NULL )
       OR
       ( _is_marked_for_approval IS NULL )
       OR
       ( _is_approved IS NULL )
       OR
       ( _is_denied IS NULL )
     ) THEN
    RETURN false;
  END IF;

  -- Mehrstufige Genehmigung aktiviert
  _is_markfa_needed := TSystem.Settings__GetBool( 'BDE_Abwesenheitsbeantragung_Mehrstufig' );

  -- Setting-Name irreführend, da ursprünglich dafür gedacht,
  -- die Sichtbarkeit von "nicht eigenen" Datensätzen auf AL zu beschränken
  -- gemeint ist für BDE und AbwBew:
  -- dass in diesem Fall die Berechtigungen auf Abteilungen eingeschränkt werden
  _is_al_restricted_rights := TSystem.Settings__GetBool( 'BDE_bdep_abteilungsleiter_rights' );


  -- Wenn nicht mehrstufig, dann wird jede AbwBew
  -- als bereits vorläufig genehmigt betrachtet
  _is_marked_fa  := ( (_is_marked_for_approval ) OR ( NOT( _is_markfa_needed ) ) );

  _is_self       := ( _usename = _llv.ll_db_usename );

  _is_supergroup :=
  (
    ( COALESCE( TSystem.roles__user__rights_personal( _usename, '' ), false ) )
    OR
    ( COALESCE( TSystem.roles__user__rights_personal( _usename, 'AbwBewill.ALohneRecht' ), false ) )
  );

  _is_finalized  := ( _is_approved OR _is_denied );


  -- Daten für den 'Editor' holen
  -- KEIN COALESCE(ll_abteilung), dies würde dazu führen dass alle ohne Abteilung (null) als eigenständige Abteilung behandelt würden
  SELECT
    true,
    llv_abteilungsleiter.ll_abteilung,
    COALESCE( llv_abteilungsleiter.ll_abteilungsleiter, false ),
    COALESCE( llv_abteilungsleiter.ll_urlaub_bewill_vorl, false ),
    COALESCE( llv_abteilungsleiter.ll_urlaub_bewill, false ),
    COALESCE( llv_abteilungsleiter.ll_urlaub_bewill_notself, false )
  FROM
    llv AS llv_abteilungsleiter
  WHERE
    ( llv_abteilungsleiter.ll_db_usename = _usename )
  INTO
    _editorexists,
    _abteilung,
    _abteilungsleiter,
    _urlaub_bewill_vorl,
    _urlaub_bewill,
    _urlaub_notself
  ;
  IF (_editorexists IS NULL) THEN
    RETURN false;
  END IF;

  -- Wenn nicht mehrstufig, dann wird dieses Recht
  -- des MA als nicht gesetzt betrachtet
  -- da in diesem Fall generell nicht vorläufig genehmigt werden darf
  _urlaub_bewill_vorl :=
  (
    ( _urlaub_bewill_vorl )
    AND
    ( _is_markfa_needed )
  );


  IF _is_al_restricted_rights THEN
    _is_same_abteilung :=
    (
      ( _llv.ll_abteilung IS NOT NULL )
      AND
      ( _abteilung IS NOT NULL )
      AND
      ( _abteilung = _llv.ll_abteilung )
    );
    IF NOT _is_same_abteilung THEN
      SELECT
        (
          EXISTS
          (
            SELECT
              llv_abteilungen.lla_abteilung
            FROM
              llv_abteilungen
            WHERE
              (
                ( _llv.ll_abteilung IS NOT NULL )
                AND
                ( llv_abteilungen.lla_abteilung = _llv.ll_abteilung )
                AND
                ( llv_abteilungen.lla_db_usename = _usename )
              )
          )
        )
      INTO
        _is_same_abteilung
      ;
    END IF;
  ELSE
    _is_same_abteilung := true;
  END IF;

  _is_al_of := ( ( _abteilungsleiter ) AND ( _is_same_abteilung ) );


  -- Im folgenden wird keine Unterscheidung bzgl. 'BDE_Abwesenheitsbeantragung_Mehrstufig'
  -- aktiv/inaktiv gemacht da '_urlaub_bewill_vorl' angepasst wird (siehe weiter oben)
  CASE _operation

    -- op: neu erstellen
    WHEN 'add' THEN
      -- Beschränkung auf wenn eines der beiden Rechte vorhanden (vorl/definitiv) oder Selbst
      -- Beschränkung auf (Wenn Setting aktiv)
      --     SYS.Personal oder SYS.Personal.AbwBewill.ALohneRecht
      --     oder
      --     Abteilungsleiter oder gleiche Abteilung (mit Berücksichtigung multipler Abteilungen - llv_abteilungen)

      RETURN
      (
        (
          -- ursprüngliches Verhalten + Rechte
          ( NOT( _is_al_restricted_rights ) )
          AND
          ( _is_self OR _urlaub_bewill_vorl OR _urlaub_bewill )
        )
        OR
        (
          -- AL beschränkt + Rechte
          ( _is_al_restricted_rights )
          AND
          (
            ( _is_self )
            OR
            (
              ( _is_supergroup OR _is_al_of OR _is_same_abteilung  )
              AND
              ( _urlaub_bewill_vorl OR _urlaub_bewill )
            )
          )
        )
      );

    -- op: bestehende löschen
    WHEN 'delete' THEN
      -- Jeder der neu anlegen darf
      -- kann dann auch wieder löschen
      -- sofern noch nicht definitiv-genehmigt/abgelehnt ODER kann definitiv-genehmigen

      SELECT
        tpersonal.bde__user_rights_operation( _usename, _llv, 'add'::TPersonal.bde__abwbew_operation, false, false, false )
      INTO
        _has_right_1
      ;

      RETURN
      (
        ( _has_right_1 )
        AND
        (
          ( NOT( _is_finalized) )
          OR
          ( _urlaub_bewill )
        )
      );

    -- op: besetehende bearbeiten
    WHEN 'edit' THEN
      -- Jeder der neu anlegen darf
      -- kann dann auch editieren
      -- sofern noch nicht definitiv-genehmigt/abgelehnt
      --     und nicht tatsächlich vorläufig-genehmigt
      --         oder kann vorläufig/definitiv-genehmigen

      SELECT
        tpersonal.bde__user_rights_operation( _usename, _llv, 'add'::TPersonal.bde__abwbew_operation, false, false, false )
      INTO
        _has_right_1
      ;

      RETURN
      (
        ( _has_right_1 )
        AND
        ( NOT( _is_finalized ) )
        AND
        (
          NOT( _is_marked_for_approval )
          OR
          ( _urlaub_bewill_vorl OR _urlaub_bewill )
        )
      );

    -- op: vorläufig genehmigen (wenn mehrstufig)
    WHEN 'markforapproval' THEN
      -- Wenn ich im gegenwärtigen Zustand editieren kann
      -- und noch nicht definitiv-genehmigt/abgelehnt
      -- und ich vorläufig genehmigen kann

      SELECT
        tpersonal.bde__user_rights_operation( _usename, _llv, 'edit'::TPersonal.bde__abwbew_operation, _is_marked_for_approval, _is_approved, _is_denied )
      INTO
        _has_right_1
      ;

      RETURN
      (
        ( _has_right_1 ) AND ( NOT( _is_finalized ) ) AND ( _urlaub_bewill_vorl )
      );

    -- op: genehmigen
    WHEN 'approve' THEN
      -- Wenn ich im gegenwärtigen Zustand editieren kann
      -- und noch nicht definitiv-genehmigt/abgelehnt
      -- und ich definitiv genehmigen kann
      -- und vorläufig genehmigen kann oder bereits (effektiv) vorläufig genehmigt

      SELECT
        tpersonal.bde__user_rights_operation( _usename, _llv, 'edit'::TPersonal.bde__abwbew_operation, _is_marked_for_approval, _is_approved, _is_denied )
      INTO
        _has_right_1
      ;

      RETURN
      (
        ( _has_right_1 )
        AND
        ( NOT( _is_finalized ) )
        AND
        ( _urlaub_bewill )
        AND
        ( _urlaub_bewill_vorl OR _is_marked_fa )
        AND
        (
          ( ( _urlaub_notself ) AND NOT(_is_self) )
          OR
          ( NOT( _urlaub_notself ) )
        )
      );

    -- op: genehmigen
    WHEN 'deny' THEN
      -- Wenn ich im gegenwärtigen Zustand vorläufig genehmigen darf
      -- oder definitiv genehmigen dürfte sofern vorläufig-genehmigt
      -- kann ich implizit auch ablehnen

      SELECT
        tpersonal.bde__user_rights_operation( _usename, _llv, 'markforapproval'::TPersonal.bde__abwbew_operation, _is_marked_for_approval, _is_approved, _is_denied )
      INTO
        _has_right_1
      ;
      SELECT
        tpersonal.bde__user_rights_operation( _usename, _llv, 'approve'::TPersonal.bde__abwbew_operation, _is_markfa_needed, _is_approved, _is_denied )
      INTO
        _has_right_2
      ;

      RETURN
      (
        ( _has_right_1 ) OR ( _has_right_2)
      );
    WHEN 'revoke' THEN
      -- Wenn ich grundsätzlich definitiv-genehmigen darf
      -- kann ich die Entscheidung auch wieder revidieren
      -- sofern bereits eine Entscheidung getroffen wurde

      SELECT
        tpersonal.bde__user_rights_operation( _usename, _llv, 'approve'::TPersonal.bde__abwbew_operation, _is_markfa_needed, false, false )
      INTO
        _has_right_1
      ;

      RETURN
      (
        ( _has_right_1 AND ( _is_finalized ) )
      );
    ELSE
      -- op: Unbekannter Wert für ENUM (vergessen zu implementieren)
      RAISE EXCEPTION 'unkown _operation enum value %', lang_text(34006);
  END CASE;
END $$ LANGUAGE plpgsql STABLE;

CREATE OR REPLACE FUNCTION tpersonal.bde__user_rights_operation(
  _usename                varchar,                         -- Benutzer welcher Zugreifen möchte
  _for_usename            varchar,                         -- Benutzer ´(seine Daten) auf welchen zugegriffen werden soll
  _operation              TPersonal.bde__abwbew_operation, -- Operation für die Recht ermittelt werden soll
  _is_marked_for_approval boolean,                         -- Bereits vorläufig genehmigt (mehrstufige Bewilligung)
  _is_approved            boolean,                         -- Bereits genehmigt
  _is_denied              boolean                          -- Bereits abgelehnt
) RETURNS boolean AS $$
DECLARE
  _right boolean;
BEGIN
  _right := false;

  if ( ( _usename IS NOT NULL ) AND ( _for_usename IS NOT NULL ) ) THEN
    SELECT
      tpersonal.bde__user_rights_operation(_usename, llv, _operation, _is_marked_for_approval, _is_approved, _is_denied)
    FROM
      llv
    WHERE
      (llv.ll_db_usename = _for_usename)
    INTO
      _right
    ;
  END IF;

  RETURN COALESCE(_right, false);
END $$ LANGUAGE plpgsql STABLE;
-- #16900 (#16771) END



-- # 16863 UPT Abwesenheiten anonymisiert
--
-- Liefert (forciert) anonymisierten bdeabgruende, ohne Rechte-Prüfung oder ähnliches
--
CREATE OR REPLACE FUNCTION tpersonal.bde__bdeabgruende_anonymize(
  IN bdeabgruende_rec  bdeabgruende,                          -- bdeabgruende Record
  IN bewill_vorl       boolean,                               -- Ist bereits vorläufig bewillligt
  IN bewill_definitiv  boolean,                               -- Ist bereits definitiv bewilligt
  IN anonym            boolean DEFAULT true,                  -- Soll Ausgabe anonymisiert werden
  OUT ab_id            integer,
  OUT ab_txt           character varying,
  OUT ab_txt_short     character varying,
  OUT ab_color         integer
  )
  RETURNS record
  AS $$
  BEGIN
    ab_id := bdeabgruende_rec.ab_id;

    IF bdeabgruende_rec.ab_anonym_txt IS NULL THEN
      ab_txt       := 'absent'; -- folgendes verursacht enorme Laufzeiten von > 2 Sekunden zB bei KB lang_text( 5126 );              -- Abwesend
      ab_txt_short := 'A'; --LEFT( ab_txt, 1 );              -- A
    ELSE
      ab_txt       := bdeabgruende_rec.ab_anonym_txt; -- Anonymisierter Text
      ab_txt_short := LEFT( ab_txt, 2 );              -- ??
    END IF;

    IF anonym THEN
      IF bewill_definitiv THEN
        ab_color := 9884566;    -- Grün
      ELSE
        IF bewill_vorl THEN
          ab_color := 10551295; -- Gelb
        ELSE
          ab_color := 14211288; -- Grau
        END IF;
      END IF;
    ELSE
      ab_txt       := bdeabgruende_rec.ab_txt;
      ab_txt_short := bdeabgruende_rec.ab_txt_short;
      ab_color     := bdeabgruende_rec.ab_color;
    END IF;
  END $$ language plpgsql STABLE PARALLEL SAFE STRICT;

--
-- Keine Funktion als DEFAULT benutzt, solche Funktionen werden eigenartigerweise MEHRFACH ausgeführt (Performance)
-- Stattdessen reversed Boolean Parameter 'nicht_anonym', damit kann für SYS_Personal/Abteilungsleiter
-- der Aufruf von llv__user_rights_personal__mine_or_abteilunsgleiter vermieden werden
--
-- Feststellen ob anonymisiert werden muss
-- Wenn
--   anonymisierten Text in 1 von 3 Farben (noch nicht / vorläufig / defintiv bewilligt)
-- Nicht
--   hinterlegten Text mit hinterlegter Farbe benutzen
--
CREATE OR REPLACE FUNCTION tpersonal.bde__bdeabgruende_anonymize_upt(
    IN  llv_rec            llv,                                   -- llv Record
    IN  bdeabgruende_rec   bdeabgruende,                          -- bdeabgruende Record
    IN  bewill_vorl        boolean,                               -- Ist bereits vorläufig bewillligt
    IN  bewill_definitiv   boolean,                               -- Ist bereits definitiv bewilligt
    IN  viewer_db_usename  varchar   DEFAULT current_user::varchar, -- ll_db_usename desjenigen, welcher die Daten betrachtet
    IN  nicht_anonym       boolean   DEFAULT false,                 -- Soll anonymisierte Ausgabe unterdrückt werden (Standard ist anonymisieren, wenn in bdeabgruende als solches markiert, dieses Flag zwingt Ausgabe als 'as is')
    IN  _user_rights_personal boolean DEFAULT null,
    OUT ab_id              integer,
    OUT ab_txt             character varying,
    OUT ab_txt_short       character varying,
    OUT ab_color           integer
    )
    RETURNS record AS $$

    DECLARE
      _do_anonymize        boolean = true;
      _has_some_rights     boolean;
      _viewerexists        boolean;
      _urlaub_bewill_vorl  boolean;
      _urlaub_bewill       boolean;
      _ll_abteilungsleiter boolean;

    BEGIN

      -- keine Eingabe / Strict. Beachte _user_rights_personal => default null, daher kein Strict mehr
      IF    llv_rec.ll_minr IS null
         OR bdeabgruende_rec.ab_id IS null
         OR viewer_db_usename IS null
      THEN
         RETURN;
      END IF;

      -- diesen IN-Parameter müsste man aus dem SSQL der UPT per macro steuern, falls bspw Mitarbeiter keine Abteilung haben (Mitarbeiter ohne Abteilung sind in beiden Fällen nicht abgedeckt und anonym);
      -- Erweiterung tpersonal.BDE__TUrlaubsplan__Get um diesen Parameter wäre erforderlich
      _do_anonymize := NOT nicht_anonym; -- nicht_anonym true >> do_anonymize auf false

      -- gar nicht erst weiter prüfen, wenn übersteuert
      IF  _do_anonymize THEN
        -- Viewer Daten holen: Hat Genehmigungsrechte?
        SELECT
          true,
          COALESCE( llv_currentuser.ll_urlaub_bewill_vorl, false ),
          COALESCE( llv_currentuser.ll_urlaub_bewill, false ),
          COALESCE( llv_currentuser.ll_abteilungsleiter, false )
        FROM
          llv AS llv_currentuser
        WHERE
          ( llv_currentuser.ll_db_usename::varchar = viewer_db_usename )
        INTO
          _viewerexists,
          _urlaub_bewill_vorl,
          _urlaub_bewill,
          _ll_abteilungsleiter
        ;

        IF ( _viewerexists IS NOT NULL ) THEN
          -- Viewer hat Genehmigungsrechte?
          _has_some_rights :=
          (
            ( _urlaub_bewill )
            OR
            (
              ( TSystem.Settings__GetBool( 'BDE_Abwesenheitsbeantragung_Mehrstufig' ) )
              AND
              ( _urlaub_bewill_vorl )
            )
          );

          -- Als anonym markieren für folgende Fälle; Beachte, wir definieren die Ausnahmen, also die Fälle, für die nicht anonym sein soll
          -- Was bedeutet Eingeschränkter Modus: https://help.prodat-erp.de/bde_berechtigungen.html
          _do_anonymize :=
          (
            ( -- Fall 1 : Eingeschränkter Modus aus: anonym, wenn kein Abteilungsleiter bzw Personalverantwortliche oder keine Genehmigungsrechte oder nicht meine Abwesenheit
                  bdeabgruende_rec.ab_anonym                                            -- Einstellung laut Abwesenheitsgründe-Tabelle
              -- folgendes = konstant
              AND NOT TSystem.Settings__GetBool( 'BDE_bdep_abteilungsleiter_rights' )   -- Eingeschränkter Modus (OFF)
              AND NOT (
                   coalesce(_user_rights_personal, TSystem.roles__user__rights_personal( viewer_db_usename, '' ))                -- Personalberechtigung
                OR _ll_abteilungsleiter                                                                          -- Abteilungsleiter irgendeiner Abteilung
                OR viewer_db_usename = llv_rec.ll_db_usename                                                     -- Meine eigene
                OR ( tpersonal.llv__user_abteilung__is_mine( viewer_db_usename, llv_rec ) AND _has_some_rights ) -- Meine Abteilung und ich darf genehmigem
              )

            )
            OR
            ( -- Fall 2 : Eingeschränkter Modus an: anonym, wenn keine Genehmigungsrechte für die Abteilung (oder personalverantwortlich) oder nicht meine Abwesenheit oder nicht meine Abteilung
                  bdeabgruende_rec.ab_anonym                                            -- Einstellung laut Abwesenheitsgründe-Tabelle
              -- folgendes = konstant
              AND TSystem.Settings__GetBool( 'BDE_bdep_abteilungsleiter_rights' )       -- Eingeschränkter Modus (ON)
              AND NOT tpersonal.bde__visibility_abwbewill( viewer_db_usename, llv_rec ) -- bewusst Eigenschaft Abteilungsleiter optional, AL wird ebenso darin geprüft
            )
          );
        END IF;
      ELSE
        _do_anonymize := true;
      END IF;

      -- Anonymisieren oder 'as is' anzeigen
      SELECT
        ( tpersonal.bde__bdeabgruende_anonymize( bdeabgruende_rec, bewill_vorl, bewill_definitiv, _do_anonymize ) ).*
      INTO
        ab_id,
        ab_txt,
        ab_txt_short,
        ab_color
      ;

    END $$ language plpgsql STABLE PARALLEL SAFE;
-- # 16863 END

CREATE OR REPLACE FUNCTION tpersonal.llv__ll_minr_update__cascade__db_link(
    IN minr_old              integer,
    IN minr_new              integer,
    IN MiNrZusammenFuehren   boolean DEFAULT false
    )
    RETURNS void
    AS $$
    BEGIN

    PERFORM * FROM dblink(tsystem.dblink__connectionstring__get(),
                          'SELECT * FROM tpersonal.llv__ll_minr_update__cascade(' || minr_old || ', ' || minr_new || ', ' || MiNrZusammenFuehren::TEXT ||  ')'
                         ) AS sub (result VARCHAR);
      RETURN;
    END $$ LANGUAGE plpgsql;

--- #19514
--- bewilligte Urlaube
CREATE OR REPLACE FUNCTION tpersonal.Urlaubsplan__bewilligte_Urlaube__Get(
    IN _viewer_db_usename    varchar,
    IN _gkonf                boolean,
    IN _mode_representative  boolean,
    IN _anfd                 date,
    IN _endd                 date,
    IN _llabt                varchar,
    IN _lang                 varchar,
    IN _minr                 integer[],
    IN _user_rights_personal boolean = null
    )
    RETURNS TABLE(
        bdab_minr                integer,
        bdab_anf                 date,
        bdab_end                 date,
        bdab_stu                 numeric(12,4),
        bdab_aus_id              integer,
        dbrid                    varchar,
        bdab_bdabb_id            integer,
        bdabb_bewill_vorl        boolean,
        bdabb_bewill             boolean,
        bdabb_ablehn             boolean,
        bdabb_beantragt          boolean,
        ad_name                  varchar,
        ad_vorn                  varchar,
        ad_fullname              varchar,
        ab_id                    integer,
        ab_txt                   character varying,
        ab_txt_short             character varying,
        ab_color                 integer,
        ab_timediff              float,
        bdabb_conflict           boolean,
        hint                     varchar,
        source                   varchar,
        abteilung                varchar,
        sortierung               integer
    )
    AS $$
    DECLARE _rec_texte record;
    BEGIN

      SELECT lang_text(25003) AS xtt25003, lang_text(981) AS xtt981, lang_text(17523) AS xtt17523, lang_text(5126) AS xttAnonymizeAbsent
        INTO _rec_texte;

      RETURN QUERY
        SELECT
               bdpb.bdab_minr,
               bdpb.bdab_anf,
               bdpb.bdab_end,
               bdpb.bdab_stu,
               bdpb.bdab_aus_id,
               bdpb.dbrid AS dbrid,
               bdpb.bdab_bdabb_id,
               false AS bdabb_bewill_vorl,
               TRUE AS bdabb_bewill,
               FALSE AS bdabb_ablehn,
               EXISTS(SELECT true FROM bdepabbe WHERE bdepabbe.bdabb_id = bdpb.bdab_bdabb_id) AS bdabb_beantragt,
               adk.ad_name,
               adk.ad_vorn,
               nameAufloesen(ll_minr) as ad_fullname,

               bde__bdeabgruende.ab_id,
               IfThen(bde__bdeabgruende.ab_txt = 'absent', _rec_texte.xttAnonymizeAbsent, bde__bdeabgruende.ab_txt) AS ab_txt,
               bde__bdeabgruende.ab_txt_short,
               bde__bdeabgruende.ab_color,

               timediff( bdpb.bdab_anf, bdpb.bdab_end, true, true )::FLOAT + 1 AS ab_timediff,

               -- Konflikte anzeigen, aber nur in der Zukunft wegen Laufzeit
               CASE WHEN bdpb.bdab_end > today() AND array_length(personal_vertreter_array, 1) > 0 THEN
                         (EXISTS(SELECT true FROM bdepab AS x WHERE (x.bdab_anf, x.bdab_end + 1) OVERLAPS (bdpb.bdab_anf, bdpb.bdab_end + 1)
                             AND x.bdab_id IS DISTINCT FROM bdpb.bdab_id
                             AND _gkonf
                             AND x.bdab_minr = ANY(personal_vertreter_array) ) --IN (SELECT /*bdpb.bdab_minr))--*/personal_Vertretungsbeziehungen(bdpb.bdab_minr)))
                           OR EXISTS(SELECT true FROM bdepabbe AS x WHERE (x.bdabb_anf, x.bdabb_end + 1) OVERLAPS (bdpb.bdab_anf, bdpb.bdab_end + 1)
                             AND x.bdabb_id IS DISTINCT FROM bdpb.bdab_bdabb_id
                             AND NOT x.bdabb_ablehn AND (NOT x.bdabb_bewill OR _gkonf)
                             AND x.bdabb_minr = ANY(personal_vertreter_array) )--IN (SELECT /*bdpb.bdab_minr))--*/personal_Vertretungsbeziehungen(bdpb.bdab_minr)))
                         )
                    ELSE
                         false
               END AS bdabb_conflict,

               -- concat_ws(' ', lang_text(25003), to_char(bdpb.modified_date,'DD.MM.YY')::VARCHAR, lang_text(981), Nameaufloesen(bdpb.modified_by), IFTHEN(bdpb.bdab_bem IS NOT NULL, '- '||lang_text(17523)||bdpb.bdab_bem,''))::VARCHAR(200) AS hint,
               concat_ws(' ', _rec_texte.xtt25003, to_char(bdpb.modified_date,'DD.MM.YY')::varchar, _rec_texte.xtt981, Nameaufloesen(bdpb.modified_by), IFTHEN(bdpb.bdab_bem IS NOT NULL, '- ' || _rec_texte.xtt17523 || bdpb.bdab_bem, ''))::varchar(200) AS hint,

               'bdepab'::varchar(10) AS source,
               ll_abteilung AS abteilung,
               10::INTEGER AS sortierung
          FROM bdepab AS bdpb
               JOIN llv ON ll_minr = bdpb.bdab_minr
               JOIN adk ON ad_krz = ll_ad_krz
               JOIN bdeabgruende ON bdeabgruende.ab_id = bdpb.bdab_aus_id
               ---
               LEFT JOIN LATERAL tpersonal.bde__bdeabgruende_anonymize_upt(llv, bdeabgruende, false, true, _viewer_db_usename, _user_rights_personal => _user_rights_personal) AS bde__bdeabgruende ON true
               LEFT JOIN LATERAL (SELECT array_agg(personal_Vertretungsbeziehungen) AS personal_vertreter_array FROM personal_Vertretungsbeziehungen(bdpb.bdab_minr)) AS personal_Vertretungsbeziehungen ON true

         WHERE
               --(bdab_anf, bdab_end + 1) OVERLAPS (CAST(:anfd AS DATE), CAST(:endd AS DATE) + 1)
               CASE
                 -- Standard
                 WHEN ll_db_usename = _viewer_db_usename OR NOT _mode_representative THEN ( bdpb.bdab_anf, bdpb.bdab_end + 1) OVERLAPS (CAST(_anfd AS DATE), CAST(_endd AS DATE) + 1)
               ELSE
                 -- aus Zeiterfassung und nicht eigener Mitarbeiter >> keine Vergangenheit und nur das aktuelle Jahr
                 ( bdpb.bdab_anf, bdpb.bdab_end + 1) OVERLAPS ( current_date - 14, CAST( _endd AS DATE ) + 1 - '11 months'::INTERVAL )
               END
               AND TPersonal.abteilung_in_liste( ll_abteilung, _llabt )
               AND NOT ll_urlaub_hidden
               AND COALESCE( ab_show, true )
               AND COALESCE( ll_endd, current_date ) >= current_date
               --AND CASE WHEN _minr IS NOT null THEN ll_minr = _minr ELSE ll_minr > 0 END
               AND CASE
                   WHEN NOT tsystem.array_null(_minr) THEN ll_minr = ANY( _minr )
                   ELSE ll_minr > 0 END
        ;

    END $$ LANGUAGE plpgsql;
---
--- Urlaubsanträge
CREATE OR REPLACE FUNCTION tpersonal.Urlaubsplan__Urlaubsantraege__Get(
    IN _viewer_db_usename    varchar,
    IN _gkonf                boolean,
    IN _mode_representative  boolean,
    IN _anfd                 date,
    IN _endd                 date,
    IN _llabt                varchar,
    IN _lang                 varchar,
    IN _minr                 integer[],
    IN _user_rights_personal boolean = null
    )
    RETURNS TABLE(
        bdab_minr                integer,
        bdab_anf                 date,
        bdab_end                 date,
        bdab_stu                 numeric(12,4),
        bdab_aus_id              integer,
        dbrid                    varchar,
        bdab_bdabb_id            integer,
        bdabb_bewill_vorl        boolean,
        bdabb_bewill             boolean,
        bdabb_ablehn             boolean,
        bdabb_beantragt          boolean,
        ad_name                  varchar,
        ad_vorn                  varchar,
        ad_fullname              varchar,
        ab_id                    integer,
        ab_txt                   character varying,
        ab_txt_short             character varying,
        ab_color                 integer,
        ab_timediff              float,
        bdabb_conflict           boolean,
        hint                     varchar,
        source                   varchar,
        abteilung                varchar,
        sortierung               integer
    )
    AS $$
    DECLARE _rec_texte record;
    BEGIN

      SELECT lang_text(12998) AS xtt12998, lang_text(981) AS xtt981, lang_text(17523) AS xtt17523, lang_text(5126) AS xttAnonymizeAbsent
        INTO _rec_texte;

      RETURN QUERY
          SELECT
                 bdabb.bdabb_minr AS bdab_minr,
                 bdabb.bdabb_anf AS bdab_anf,
                 bdabb.bdabb_end AS bdab_end,
                 bdabb.bdabb_stu AS bdab_stu,
                 bdabb.bdabb_aus_id AS bdab_aus_id,
                 bdabb.dbrid AS dbrid,
                 --NULL AS bdab_bdabb_id,
                 null::integer AS bdab_bdabb_id,
                 bdabb.bdabb_bewill_vorl,
                 bdabb.bdabb_bewill,
                 bdabb.bdabb_ablehn,
                 true AS bdabb_beantragt,
                 adk.ad_name,
                 adk.ad_vorn,
                 nameAufloesen(ll_minr) as ad_fullname,

                 bde__bdeabgruende.ab_id,
                 IfThen(bde__bdeabgruende.ab_txt = 'absent', _rec_texte.xttAnonymizeAbsent, bde__bdeabgruende.ab_txt) AS ab_txt,
                 bde__bdeabgruende.ab_txt_short,
                 bde__bdeabgruende.ab_color,

                 timediff(bdabb_anf,bdabb_end,true,true)::FLOAT+1 AS ab_timediff,

                 CASE WHEN bdabb.bdabb_anf > today() AND array_length(personal_vertreter_array, 1) > 0 THEN
                     (EXISTS(SELECT true FROM bdepab AS x WHERE (x.bdab_anf, x.bdab_end + 1) OVERLAPS (bdabb.bdabb_anf, bdabb.bdabb_end + 1)
                         AND x.bdab_bdabb_id IS DISTINCT FROM bdabb.bdabb_id
                         AND (NOT bdabb.bdabb_bewill OR _gkonf)
                         AND x.bdab_minr = ANY(personal_vertreter_array) )--IN (SELECT personal_Vertretungsbeziehungen(bdabb.bdabb_minr)))
                       OR EXISTS(SELECT true FROM bdepabbe AS x WHERE (x.bdabb_anf, x.bdabb_end + 1) OVERLAPS (bdabb.bdabb_anf, bdabb.bdabb_end + 1)
                         AND x.bdabb_id IS DISTINCT FROM bdabb.bdabb_id
                         AND NOT x.bdabb_ablehn AND (NOT x.bdabb_bewill OR _gkonf)
                         AND x.bdabb_minr = ANY(personal_vertreter_array) )--IN (SELECT personal_Vertretungsbeziehungen(bdabb.bdabb_minr)))
                     )
                    ELSE
                         false
                 END AS bdabb_conflict,

                 --concat_ws(' ', lang_text(12998), to_char(bdabb.modified_date,'DD.MM.YY')::varchar, lang_text(981), Nameaufloesen(bdabb.modified_by), IFTHEN(bdabb.bdabb_bem IS NOT NULL, '- '||lang_text(17523) || bdabb.bdabb_bem, ''))::varchar(200) AS hint,
                 concat_ws(' ', _rec_texte.xtt12998, to_char(bdabb.modified_date,'DD.MM.YY')::varchar, _rec_texte.xtt981, Nameaufloesen(bdabb.modified_by), IFTHEN(bdabb.bdabb_bem IS NOT NULL, '- ' || _rec_texte.xtt17523 || bdabb.bdabb_bem, ''))::varchar(200) AS hint,

                 'bdepabbe'::varchar(10) AS source,
                 ll_abteilung AS abteilung,
                 10::INTEGER AS sortierung
            FROM bdepabbe AS bdabb
                 JOIN llv ON ll_minr = bdabb_minr
                 JOIN adk ON ad_krz = ll_ad_krz
                 JOIN bdeabgruende ON bdeabgruende.ab_id = bdabb_aus_id
                 LEFT JOIN LATERAL tpersonal.bde__bdeabgruende_anonymize_upt(llv, bdeabgruende, bdabb.bdabb_bewill_vorl, bdabb.bdabb_bewill, _viewer_db_usename, _user_rights_personal => _user_rights_personal) AS bde__bdeabgruende ON true
                 LEFT JOIN LATERAL (SELECT array_agg(personal_Vertretungsbeziehungen) AS personal_vertreter_array FROM personal_Vertretungsbeziehungen(bdabb.bdabb_minr)) AS personal_Vertretungsbeziehungen ON true
           WHERE
                 -- (bdabb_anf, bdabb_end + 1) OVERLAPS (CAST(:anfd AS DATE), CAST(:endd AS DATE) + 1)
                 CASE
                   -- Standard
                   WHEN ll_db_usename = _viewer_db_usename OR NOT _mode_representative THEN (bdabb_anf, bdabb_end + 1) OVERLAPS (CAST(_anfd AS DATE), CAST(_endd AS DATE) + 1)
                 ELSE
                   -- aus Zeiterfassung und nicht eigener Mitarbeiter >> keine Vergangenheit und nur das aktuelle Jahr
                   (bdabb_anf, bdabb_end + 1) OVERLAPS ( current_date - 14, CAST(_endd AS DATE) + 1 - '11 months'::INTERVAL )
                 END
                 AND TPersonal.abteilung_in_liste(ll_abteilung, _llabt)
                 AND NOT ll_urlaub_hidden
                 AND COALESCE(ab_show, true)
                 AND COALESCE(ll_endd, current_date) >= current_date
                 AND NOT bdabb.bdabb_ablehn
                 AND NOT EXISTS(SELECT true FROM bdepab WHERE bdepab.bdab_bdabb_id = bdabb_id)
                 --AND CASE WHEN _minr IS NOT null THEN ll_minr = _minr ELSE ll_minr > 0 END
                 AND CASE
                   WHEN NOT tsystem.array_null(_minr) THEN ll_minr = ANY( _minr )
                   ELSE ll_minr > 0 END
                 ;

    END $$ LANGUAGE plpgsql;
---
--- Schulungsmaßnahmen
CREATE OR REPLACE FUNCTION tpersonal.Urlaubsplan__Schulungsmassnahmen__Get(
    IN _viewer_db_usename    varchar,
    IN _gkonf                boolean,
    IN _mode_representative  boolean,
    IN _anfd                 date,
    IN _endd                 date,
    IN _llabt                varchar,
    IN _lang                 varchar,
    IN _minr                 integer[]
    )
    RETURNS TABLE(
        bdab_minr                integer,
        bdab_anf                 date,
        bdab_end                 date,
        bdab_stu                 numeric(12,4),
        bdab_aus_id              integer,
        dbrid                    varchar,
        bdab_bdabb_id            integer,
        bdabb_bewill_vorl        boolean,
        bdabb_bewill             boolean,
        bdabb_ablehn             boolean,
        bdabb_beantragt          boolean,
        ad_name                  varchar,
        ad_vorn                  varchar,
        ad_fullname              varchar,
        ab_id                    integer,
        ab_txt                   character varying,
        ab_txt_short             character varying,
        ab_color                 integer,
        ab_timediff              float,
        bdabb_conflict           boolean,
        hint                     varchar,
        source                   varchar,
        abteilung                varchar,
        sortierung               integer
    )
    AS $$
    BEGIN

    RETURN QUERY
          SELECT
            llm_minr AS bdab_minr,
            date(dat_begin) AS bdab_anf,
            date(dat_end) AS bdab_end,
            null::numeric(12,4) AS bdab_stu,
            bdeabgruende.ab_id AS bdab_aus_id,
            skillplan.dbrid AS dbrid,
            null::integer AS bdab_bdabb_id,
         --   llm_status_id AS bdabb_bewill,
            false AS bdabb_bewill_vorl,
            IFTHEN(skstatus = 'D', True, False) AS bdabb_bewill,
            FALSE AS bdabb_ablehn,
            FALSE AS bdabb_beantragt,
            adk.ad_name,
            adk.ad_vorn,
            nameAufloesen(ll_minr) as ad_fullname,

            bde__bdeabgruende.ab_id,
            bde__bdeabgruende.ab_txt,
            bde__bdeabgruende.ab_txt_short,
            bde__bdeabgruende.ab_color,

            timediff(date(dat_begin),date(dat_end),true,true)::FLOAT+1 AS ab_timediff,
            FALSE AS bdabb_conflict,
            concat_ws('', lang_artbez(skp_ak_nr, _lang), ': ', dat_subject, ' (', dat_location, ')', E'\n', IFTHEN(skstatus = 'D', lang_text(15543),'') )::VARCHAR(200) AS hint,    --wenn Status 'D', dann Genehmigt
            'skillplan'::VARCHAR(10) AS source,
            ll_abteilung AS abteilung,
            10::INTEGER AS sortierung
          FROM skillplan
            JOIN llv_members ON llm_tablename = dat_tablename AND llm_parent_dbrid IN (SELECT skillplan.dbrid FROM skillplan s0 WHERE s0.dat_id=skillplan.dat_id OR s0.dat_id=skillplan.dat_altdat_for_dat_id)
            JOIN llv ON ll_minr = llm_minr
            JOIN adk ON ad_krz = ll_ad_krz
            JOIN bdeabgruende ON bdeabgruende.ab_id = 108  -- 108=Schulungsmaßnahmen
            LEFT JOIN LATERAL( SELECT TRecnoParam.GetEnum('skillplan.status.plan', skillplan.dbrid) AS skstatus ) AS sk ON TRUE  -- Status der Schulung
            ---
            LEFT JOIN LATERAL tpersonal.bde__bdeabgruende_anonymize_upt(llv, bdeabgruende, false, true, _viewer_db_usename) AS bde__bdeabgruende ON true
          WHERE TPersonal.abteilung_in_liste(ll_abteilung, _llabt)
            AND COALESCE(ll_endd, current_date) >= current_date
            AND NOT ll_urlaub_hidden
            AND dat_begin IS NOT NULL
            --AND CASE WHEN _minr IS NOT null THEN ll_minr = _minr ELSE ll_minr > 0 END
            AND CASE
                   WHEN NOT tsystem.array_null(_minr) THEN ll_minr = ANY( _minr )
                   ELSE ll_minr > 0 END
            ;

    END $$ LANGUAGE plpgsql;
---
--- Abwesend
CREATE OR REPLACE FUNCTION tpersonal.Urlaubsplan__Abwesend__Get(
    IN _viewer_db_usename    varchar,
    IN _gkonf                boolean,
    IN _mode_representative  boolean,
    IN _anfd                 date,
    IN _endd                 date,
    IN _llabt                varchar,
    IN _lang                 varchar,
    IN _minr                 integer[]
    )
    RETURNS TABLE(
        bdab_minr                integer,
        bdab_anf                 date,
        bdab_end                 date,
        bdab_stu                 numeric(12,4),
        bdab_aus_id              integer,
        dbrid                    varchar,
        bdab_bdabb_id            integer,
        bdabb_bewill_vorl        boolean,
        bdabb_bewill             boolean,
        bdabb_ablehn             boolean,
        bdabb_beantragt          boolean,
        ad_name                  varchar,
        ad_vorn                  varchar,
        ad_fullname              varchar,
        ab_id                    integer,
        ab_txt                   character varying,
        ab_txt_short             character varying,
        ab_color                 integer,
        ab_timediff              float,
        bdabb_conflict           boolean,
        hint                     varchar,
        source                   varchar,
        abteilung                varchar,
        sortierung               integer
    )
    AS $$
    BEGIN

    RETURN QUERY
          SELECT
            mitpln.mpl_minr AS bdab_minr,
            mitpln.mpl_date AS bdab_anf,
            mitpln.mpl_date AS bdab_end,
            null::numeric(12,4) AS bdab_stu,
            11111 AS bdab_aus_id, --Pflicht für die Oberfläche
            mitpln.dbrid AS dbrid,
            null::integer AS bdab_bdabb_id,
            false AS bdabb_bewill_vorl,
            true AS bdabb_bewill,
            false AS bdabb_ablehn,
            false AS bdabb_beantragt,
            null::varchar AS ad_name,
            null::varchar AS ad_vorn,
            null::varchar AS ad_fullname, --wegen Performance hier entfernt   --ad_name, ad_vorn, nameAufloesen(mpl_minr) as ad_fullname,
            null::integer AS ab_id,
            lang_text(2117) AS ab_txt,
            'A'::varchar AS ab_txt_short, --Anwesend ~ 'a' Schriftart Marlett erzeugt Häkchen über die Oberfläche
            null::integer AS ab_color,
            null::float AS ab_timediff,
            false AS bdabb_conflict,
            null::varchar AS hint,

            'mitpln'::varchar(10) AS source,
            ll_abteilung AS abteilung,
            10::integer AS sortierung
          FROM mitpln
            JOIN llv ON ll_minr = mpl_minr
          WHERE
            TPersonal.abteilung_in_liste(ll_abteilung, _llabt)
            -- hier abweichend wegen Performance, nicht der volle Zeitraum
            AND mpl_date BETWEEN current_date - 14 AND current_date - 1
            -- AND mpl_saldo>0
            AND (mpl_saldo = 0 AND mpl_absaldo = 0 AND mpl_min > 0)
            -- keine eingetragene Abwesenheit (zB Stundenausgleich 0)
            AND NOT EXISTS(SELECT true FROM bdepab WHERE bdepab.bdab_minr = mpl_minr AND mpl_date BETWEEN bdepab.bdab_anf AND bdepab.bdab_end)
            AND (ll_endd IS null OR ll_endd >= current_date)
            --AND CASE WHEN _minr IS NOT null THEN ll_minr = _minr ELSE ll_minr > 0 END
            AND CASE
                   WHEN NOT tsystem.array_null(_minr) THEN ll_minr = ANY( _minr )
                   ELSE ll_minr > 0 END
            ;

    END $$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION tpersonal.Urlaubsplan__Abwesend__Get(
    IN _viewer_db_usename    varchar,
    IN _gkonf                boolean,
    IN _mode_representative  boolean,
    IN _anfd                 date,
    IN _endd                 date,
    IN _llabt                varchar,
    IN _lang                 varchar,
    IN _minr                 integer
    )
    RETURNS TABLE(
        bdab_minr                integer,
        bdab_anf                 date,
        bdab_end                 date,
        bdab_stu                 numeric(12,4),
        bdab_aus_id              integer,
        dbrid                    varchar,
        bdab_bdabb_id            integer,
        bdabb_bewill_vorl        boolean,
        bdabb_bewill             boolean,
        bdabb_ablehn             boolean,
        bdabb_beantragt          boolean,
        ad_name                  varchar,
        ad_vorn                  varchar,
        ad_fullname              varchar,
        ab_id                    integer,
        ab_txt                   character varying,
        ab_txt_short             character varying,
        ab_color                 integer,
        ab_timediff              float,
        bdabb_conflict           boolean,
        hint                     varchar,
        source                   varchar,
        abteilung                varchar,
        sortierung               integer
    )
    AS $$
    BEGIN

        RETURN QUERY
        SELECT * FROM tpersonal.Urlaubsplan__Abwesend__Get(
             _viewer_db_usename
            ,_gkonf
            ,_mode_representative
            ,_anfd
            ,_endd
            ,_llabt
            ,_lang
            ,tsystem.array__create(_minr)
          );

    END $$ LANGUAGE plpgsql;
---
--- Pseudoeintrag für Mitarbeiter ohne Urlaub
CREATE OR REPLACE FUNCTION tpersonal.Urlaubsplan__Eintrag_ohne_Urlaub__Get(
    IN _viewer_db_usename    varchar,
    IN _gkonf                boolean,
    IN _mode_representative  boolean,
    IN _anfd                 date,
    IN _endd                 date,
    IN _llabt                varchar,
    IN _lang                 varchar,
    IN _minr                 integer[]
    )
    RETURNS TABLE(
        bdab_minr                integer,
        bdab_anf                 date,
        bdab_end                 date,
        bdab_stu                 numeric(12,4),
        bdab_aus_id              integer,
        dbrid                    varchar,
        bdab_bdabb_id            integer,
        bdabb_bewill_vorl        boolean,
        bdabb_bewill             boolean,
        bdabb_ablehn             boolean,
        bdabb_beantragt          boolean,
        ad_name                  varchar,
        ad_vorn                  varchar,
        ad_fullname              varchar,
        ab_id                    integer,
        ab_txt                   varchar,
        ab_txt_short             varchar,
        ab_color                 integer,
        ab_timediff              float,
        bdabb_conflict           boolean,
        hint                     varchar,
        source                   varchar,
        abteilung                varchar,
        sortierung               integer
    )
    AS $$
    BEGIN

    RETURN QUERY
          SELECT
            ll_minr AS bdab_minr,
            null::date AS bdab_anf,
            null::date AS bdab_end,
            null::numeric(12,4) AS bdab_stu,
            null::integer AS bdab_aus_id,
            llv.dbrid AS dbrid,
            null::integer AS bdab_bdabb_id,
            false AS bdabb_bewill_vorl,
            false AS bdabb_bewill,
            false AS bdabb_ablehn,
            false AS bdabb_beantragt,
            adk.ad_name,
            adk.ad_vorn,
            nameAufloesen(ll_minr) as ad_fullname,
            null::integer AS ab_id,
            null::varchar AS ab_txt,
            null::varchar AS ab_txt_short,
            null::integer AS ab_color,
            null::float AS ab_timediff,
            false AS bdabb_conflict,
            null::varchar AS hint,

            'llv'::VARCHAR(10) AS source,
            ll_abteilung AS abteilung,
            10::INTEGER AS sortierung
          FROM llv
            JOIN adk ON ad_krz = ll_ad_krz
          WHERE TPersonal.abteilung_in_liste(ll_abteilung, _llabt)
            AND COALESCE(ll_endd, current_date) >= current_date
            AND NOT ll_urlaub_hidden

            AND NOT EXISTS(SELECT true FROM bdepab   WHERE bdepab.bdab_minr  = ll_minr)
            AND NOT EXISTS(SELECT true FROM bdepabbe WHERE bdepabbe.bdabb_minr = ll_minr)
            --AND CASE WHEN _minr IS NOT null THEN ll_minr = _minr ELSE ll_minr > 0 END
            AND CASE
                   WHEN NOT tsystem.array_null(_minr) THEN ll_minr = ANY( _minr )
                   ELSE ll_minr > 0 END
            ;

    END $$ LANGUAGE plpgsql;
---
--- Leerzeile
CREATE OR REPLACE FUNCTION tpersonal.Urlaubsplan__Leerzeile__Get(
    IN _viewer_db_usename    varchar,
    IN _gkonf                boolean,
    IN _mode_representative  boolean,
    IN _anfd                 date,
    IN _endd                 date,
    IN _llabt                varchar
    --IN _lang                 varchar,
    --IN _minr                 integer = null
    )
    RETURNS TABLE(
        bdab_minr                integer,
        bdab_anf                 date,
        bdab_end                 date,
        bdab_stu                 numeric(12,4),
        bdab_aus_id              integer,
        dbrid                    varchar,
        bdab_bdabb_id            integer,
        bdabb_bewill_vorl        boolean,
        bdabb_bewill             boolean,
        bdabb_ablehn             boolean,
        bdabb_beantragt          boolean,
        ad_name                  varchar,
        ad_vorn                  varchar,
        ad_fullname              varchar,
        ab_id                    integer,
        ab_txt                   varchar,
        ab_txt_short             varchar,
        ab_color                 integer,
        ab_timediff              float,
        bdabb_conflict           boolean,
        hint                     varchar,
        source                   varchar,
        abteilung                varchar,
        sortierung               integer
    )
    AS $$
    BEGIN

    RETURN QUERY
          SELECT
            null::integer AS bdab_minr,
            null::date AS bdab_anf,
            null::date AS bdab_end,
            null::numeric(12,4) AS bdab_stu,
            null::integer AS bdab_aus_id,
            null::varchar AS dbrid,
            null::integer AS bdab_bdabb_id,
            false AS bdabb_bewill_vorl,
            false AS bdabb_bewill,
            false AS bdabb_ablehn,
            false AS bdabb_beantragt,
            null::varchar,
            null::varchar,
            ''::varchar as ad_fullname,
            null::integer AS ab_id,
            null::varchar AS ab_txt,
            null::varchar AS ab_txt_short,
            null::integer AS ab_color,
            null::float AS ab_timediff,
            false AS bdabb_conflict,
            null::varchar AS hint,

            null::varchar AS source,
            ll_abteilung::varchar AS abteilung,
            20::integer AS sortierung
          FROM (SELECT unnest(string_to_array(_llabt, ',')) as ll_abteilung) as leer
          WHERE _llabt LIKE '%,%'
          ;

    END $$ LANGUAGE plpgsql;
---
--- Abteilungszeile
CREATE OR REPLACE FUNCTION tpersonal.Urlaubsplan__Abteilungszeile__Get(
    IN _viewer_db_usename    varchar,
    IN _gkonf                boolean,
    IN _mode_representative  boolean,
    IN _anfd                 date,
    IN _endd                 date,
    IN _llabt                varchar
    --IN _lang                 varchar,
    --IN _minr                 integer = null
    )
    RETURNS TABLE(
        bdab_minr                integer,
        bdab_anf                 date,
        bdab_end                 date,
        bdab_stu                 numeric(12,4),
        bdab_aus_id              integer,
        dbrid                    varchar,
        bdab_bdabb_id            integer,
        bdabb_bewill_vorl        boolean,
        bdabb_bewill             boolean,
        bdabb_ablehn             boolean,
        bdabb_beantragt          boolean,
        ad_name                  varchar,
        ad_vorn                  varchar,
        ad_fullname              varchar,
        ab_id                    integer,
        ab_txt                   varchar,
        ab_txt_short             varchar,
        ab_color                 integer,
        ab_timediff              float,
        bdabb_conflict           boolean,
        hint                     varchar,
        source                   varchar,
        abteilung                varchar,
        sortierung               integer
    )
    AS $$
    BEGIN

    RETURN QUERY
          SELECT
            null::integer AS bdab_minr,
            null::date AS bdab_anf,
            null::date AS bdab_end,
            null::numeric(12,4) AS bdab_stu,
            null::integer AS bdab_aus_id,
            null::varchar AS dbrid,
            null::integer AS bdab_bdabb_id,
            false AS bdabb_bewill_vorl,
            false AS bdabb_bewill,
            false AS bdabb_ablehn,
            false AS bdabb_beantragt,
            null::varchar,
            null::varchar,
            ''::varchar as ad_fullname,
            null::integer AS ab_id,
            null::varchar AS ab_txt,
            null::varchar AS ab_txt_short,
            null::integer AS ab_color,
            null::float AS ab_timediff,
            false AS bdabb_conflict,
            null::varchar AS hint,

            null::varchar AS source,
            ll_abteilung::varchar AS abteilung,
            0::integer AS sortierung
          FROM (SELECT unnest(string_to_array(_llabt, ',')) as ll_abteilung) as leer
          WHERE _llabt LIKE '%,%'
        ;

    END $$ LANGUAGE plpgsql;
---
--- Funktion aktualisiert Query für Urlaubsplan. Mit Eingangsparameter _minr kommt Query nur für ein Mitarbeiter
CREATE OR REPLACE FUNCTION tpersonal.BDE__TUrlaubsplan__Get(
    IN _viewer_db_usename    varchar,
    IN _gkonf                boolean,
    IN _mode_representative  boolean,
    IN _anfd                 date,
    IN _endd                 date,
    IN _llabt                varchar,
    IN _lang                 varchar,
    IN _minr                 integer[]
    )
    RETURNS TABLE(
        bdab_minr                integer,
        bdab_anf                 date,
        bdab_end                 date,
        bdab_stu                 numeric(12,4),
        bdab_aus_id              integer,
        dbrid                    varchar,
        bdab_bdabb_id            integer,
        bdabb_bewill_vorl        boolean,
        bdabb_bewill             boolean,
        bdabb_ablehn             boolean,
        bdabb_beantragt          boolean,
        ad_name                  varchar,
        ad_vorn                  varchar,
        ad_fullname              varchar,
        ab_id                    integer,
        ab_txt                   varchar,
        ab_txt_short             varchar,
        ab_color                 integer,
        ab_timediff              float,
        bdabb_conflict           boolean,
        hint                     varchar,
        source                   varchar,
        abteilung                varchar,
        sortierung               integer
    )
    AS $$
    DECLARE
        _user_rights_personal      boolean;
    BEGIN

        -- initiale globale einmalige Auswertungen zur Weitergabe zur Vermeidung SubSelects
        _user_rights_personal := TSystem.roles__user__rights_personal( _viewer_db_usename, '' );

        RETURN QUERY
        --- bewilligte Urlaube
        SELECT * FROM tpersonal.Urlaubsplan__bewilligte_Urlaube__Get( _viewer_db_usename, _gkonf, _mode_representative, _anfd, _endd, _llabt, _lang, _minr, _user_rights_personal )
        UNION
        --- Urlaubsanträge
        SELECT * FROM tpersonal.Urlaubsplan__Urlaubsantraege__Get( _viewer_db_usename, _gkonf, _mode_representative, _anfd, _endd, _llabt, _lang, _minr, _user_rights_personal )
        UNION
        --- Schulungsmaßnahmen
        SELECT * FROM tpersonal.Urlaubsplan__Schulungsmassnahmen__Get( _viewer_db_usename, _gkonf, _mode_representative, _anfd, _endd, _llabt, _lang, _minr )
        UNION
        --- Abwesend
        SELECT * FROM tpersonal.Urlaubsplan__Abwesend__Get( _viewer_db_usename, _gkonf, _mode_representative, _anfd, _endd, _llabt, _lang, _minr )
        UNION
        --- Pseudoeintrag für Mitarbeiter ohne Urlaub
        SELECT * FROM tpersonal.Urlaubsplan__Eintrag_ohne_Urlaub__Get( _viewer_db_usename, _gkonf, _mode_representative, _anfd, _endd, _llabt, _lang, _minr )
        UNION
        --- Leerzeile
        SELECT * FROM tpersonal.Urlaubsplan__Leerzeile__Get( _viewer_db_usename, _gkonf, _mode_representative, _anfd, _endd, _llabt ) --, _lang, _minr
        UNION
        --- Abteilungszeile
        SELECT * FROM tpersonal.Urlaubsplan__Abteilungszeile__Get( _viewer_db_usename, _gkonf, _mode_representative, _anfd, _endd, _llabt ) --, _lang, _minr
        ;
    END $$ LANGUAGE plpgsql;

--- Funktion aktualisiert Query für Urlaubsplan. Mit Eingangsparameter _minr kommt Query nur für ein Mitarbeiter
CREATE OR REPLACE FUNCTION tpersonal.BDE__TUrlaubsplan__Get(
    IN _viewer_db_usename    varchar,
    IN _gkonf                boolean,
    IN _mode_representative  boolean,
    IN _anfd                 date,
    IN _endd                 date,
    IN _llabt                varchar,
    IN _lang                 varchar,
    IN _minr                 integer
    )
    RETURNS TABLE(
        bdab_minr                integer,
        bdab_anf                 date,
        bdab_end                 date,
        bdab_stu                 numeric(12,4),
        bdab_aus_id              integer,
        dbrid                    varchar,
        bdab_bdabb_id            integer,
        bdabb_bewill_vorl        boolean,
        bdabb_bewill             boolean,
        bdabb_ablehn             boolean,
        bdabb_beantragt          boolean,
        ad_name                  varchar,
        ad_vorn                  varchar,
        ad_fullname              varchar,
        ab_id                    integer,
        ab_txt                   varchar,
        ab_txt_short             varchar,
        ab_color                 integer,
        ab_timediff              float,
        bdabb_conflict           boolean,
        hint                     varchar,
        source                   varchar,
        abteilung                varchar,
        sortierung               integer
    )
    AS $$
    BEGIN

        RETURN QUERY
        SELECT * FROM tpersonal.BDE__TUrlaubsplan__Get(
             _viewer_db_usename
            ,_gkonf
            ,_mode_representative
            ,_anfd
            ,_endd
            ,_llabt
            ,_lang
            ,tsystem.array__create(_minr)
        );

    END $$ LANGUAGE plpgsql;
--- #19514 END

---#21205 Mitarbeiternummer ändern im Mitarbeiterverzeichnis erweitern um Änderung an Logins zu ermöglichen
CREATE OR REPLACE FUNCTION tpersonal.llv__ll_db_usename_update__cascade(
        IN db_usename_old               varchar,
        IN db_usename_new               varchar,
        IN db_usename_ZusammenFuehren   boolean DEFAULT false
    ) RETURNS VOID AS $$

  BEGIN
    -- Audit-Log abschalten: so tun, als wären wir 'syncro' (u.a. wg. modified-Triggern)
    --- Solte nicht die Änderung geloggt werden?
    SET LOCAL SESSION AUTHORIZATION 'syncro';


    UPDATE ab2                   SET a2_v_ll_dbusename        = db_usename_new WHERE a2_v_ll_dbusename        = db_usename_old;
    UPDATE adkzust               SET azust_ll_db_usename      = db_usename_new WHERE azust_ll_db_usename      = db_usename_old;

    IF db_usename_ZusammenFuehren                                                        --- beim zussamführen darf nur ein Artikelzustädige sein
          AND exists( SELECT true FROM artzust WHERE az_ll_db_usename = db_usename_old)
          AND exists( SELECT true FROM artzust WHERE az_ll_db_usename = db_usename_new)
      THEN
              DELETE FROM artzust WHERE az_ll_db_usename = db_usename_old;
    END IF;
    UPDATE artzust               SET az_ll_db_usename         = db_usename_new WHERE az_ll_db_usename         = db_usename_old;

    UPDATE tsystem_wawi.beleg_k__extend_wawi_ansprechpartner
                                 SET k_apint_db_usename1      = db_usename_new,
                                     k_apint_db_usename2      = db_usename_new,
                                     k_pruef_db_usename1      = db_usename_new,
                                     k_pruef_db_usename1_real = db_usename_new WHERE k_pruef_db_usename1_real = db_usename_old;
    UPDATE tsystem_wawi.beleg_p  SET p_apint_db_usename       = db_usename_new WHERE p_apint_db_usename       = db_usename_old;
    UPDATE bestvorschlagpos      SET bvp_db_usename           = db_usename_new WHERE bvp_db_usename           = db_usename_old;
    UPDATE lagerlog              SET lo_user                  = db_usename_new WHERE lo_user                  = db_usename_old;
    UPDATE ldsdok                SET ld_ltauser               = db_usename_new WHERE ld_ltauser               = db_usename_old;

    IF db_usename_ZusammenFuehren                                                 --- beim zussamführen darf nur ein user sein
        AND exists( SELECT true FROM llv WHERE ll_db_usename = db_usename_old)
        AND exists( SELECT true FROM llv WHERE ll_db_usename = db_usename_new)
      THEN
            DELETE FROM llv WHERE ll_db_usename = db_usename_old;
    END IF;

    IF NOT db_usename_ZusammenFuehren THEN
        UPDATE llv               SET ll_db_usename            = db_usename_new WHERE ll_db_usename            = db_usename_old;
    END IF;

    UPDATE tlog.auditlog         SET l_user                   = db_usename_new WHERE l_user                   = db_usename_old;

    UPDATE llv_abteilungen       SET lla_db_usename           = db_usename_new WHERE lla_db_usename           = db_usename_old;
    UPDATE tsystem.log           SET log_user                 = db_usename_new WHERE log_user                 = db_usename_old;
    UPDATE mainmenurights        SET mmr_username             = db_usename_new WHERE mmr_username             = db_usename_old;
    UPDATE msgrcv                SET msgrcv_user              = db_usename_new WHERE msgrcv_user              = db_usename_old;
    UPDATE op2                   SET o2_v_ll_dbusename        = db_usename_new WHERE o2_v_ll_dbusename        = db_usename_old;
    UPDATE recnocomments         SET rc_user                  = db_usename_new WHERE rc_user                  = db_usename_old;
    UPDATE userbuttons           SET usrbt_user               = db_usename_new WHERE usrbt_user               = db_usename_old;
    UPDATE userroles             SET uro_db_usename           = db_usename_new WHERE uro_db_usename           = db_usename_old;

    IF db_usename_ZusammenFuehren THEN
      DELETE FROM llv WHERE ll_db_usename = db_usename_old;
    END IF;

    RESET SESSION AUTHORIZATION;

    RETURN;
 END $$ LANGUAGE plpgsql;
---
CREATE OR REPLACE FUNCTION tpersonal.llv__ll_db_usename_update__cascade__db_link(
    IN db_usename_old               varchar,
    IN db_usename_new               varchar,
    IN db_usename_ZusammenFuehren   boolean DEFAULT false
    )
    RETURNS void
    AS $$
    BEGIN

    PERFORM * FROM dblink(tsystem.dblink__connectionstring__get(),
                          'SELECT * FROM tpersonal.llv__ll_db_usename_update__cascade(' || quote_literal( db_usename_old ) || ', ' || quote_literal( db_usename_new ) || ', ' || db_usename_ZusammenFuehren::TEXT ||  ')'
                         ) AS sub (result VARCHAR);

    RETURN;
 END $$ LANGUAGE plpgsql;
---
CREATE OR REPLACE FUNCTION tpersonal.llv__ll_db_usename_update__rtf_29320(    ---   llv__ident_minr_or_usename__update
        IN _cb_minr_rename integer,
        IN _minr_alt       varchar,
      IN _minr_neu       varchar
    ) RETURNS void AS $$
 BEGIN

    IF _cb_minr_rename = 0 THEN
        PERFORM * FROM tpersonal.llv__ll_minr_update__cascade__db_link( minr_old => _minr_alt::integer, minr_new => _minr_neu::integer, MiNrZusammenFuehren =>false );
    ELSIF _cb_minr_rename = 1 THEN
        PERFORM * FROM tpersonal.llv__ll_db_usename_update__cascade__db_link( db_usename_old => _minr_alt, db_usename_new => _minr_neu, db_usename_ZusammenFuehren => false );
    ELSIF _cb_minr_rename = 2 THEN
        PERFORM  * FROM tpersonal.llv__ll_db_usename_update__cascade__db_link( db_usename_old => _minr_alt, db_usename_new => _minr_neu, db_usename_ZusammenFuehren => true );
    END IF;

 END $$ LANGUAGE plpgsql STABLE;